package dareka.processor.impl;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.FutureTask;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import dareka.common.LRUMap;
import dareka.common.CloseUtil;
import dareka.common.Logger;
import dareka.common.M3u8Util;
import dareka.common.Pair;
import dareka.extensions.SystemEventListener;
import dareka.processor.HttpHeader;
import dareka.processor.HttpRequestHeader;
import dareka.processor.HttpResponseHeader;
import dareka.processor.Processor;
import dareka.processor.Resource;
import dareka.processor.StringResource;
import dareka.processor.TransferListener;
import dareka.processor.URLResource;

// - 2024-08-08: 仕様変更でキャッシュが出来なくなった.
// - 本格的に対応するには時間が必要であるため、簡易的な実装をまず行なう.
// - 選択できるうちで最も品質がよいモード(例: 720p, 1080p)を選択した場合だけキャッシュ保存を
//   する処理をまず実装する.
// - そのために変更した部分にコメントで【縮退実装2024-08-08】と書く.

// - 2023年下旬に導入されたAWSから配信されるCommon Media Application Format形式.
//   2024年2月20日から本格稼動した. DMC-HLSの次の配信方式. .
//   開発部の名前でDMS、通信サブドメインでDomandとも呼ぶ.
//   (DMC(DWANGO Media Cluster), DMS(DWANGO Media Services))
// - この配信はチャンク暗号化されているから、部分的なキャッシュを挟めない.
//   完全なキャッシュがある場合にのみプレイリスト自体を乗っ取り、キャッシュを利用する.
// - DMC-HLSと同じくwatchページに画質一覧情報がある. それらはWatchVars.javaが処理する.
// - smXXXとリソースのURLとの対応を得る綺麗な方法が無い.
//   javascriptで通信関数をadviceしてsmXXXをブラウザからnicocache側に伝える.
// - キャッシュ保存時はニコ動サーバーの各種URLを使うが、キャッシュ利用時に
//   MasterPlaylist以外ではNicocacheローカルなURLに置き換えてクライアントへ渡す.
//   この設計の目的はDMC-HLSキャッシュをCmafCachingProcessorが利用出来るようにす
//   るため. (キャッシュディレクトリにmaster.m3u8を持ってさえいれば、兄弟ファイ
//   ルでも子孫ファイルでも任意の構造を扱えるようにするため)

//public class CmafCachingProcessor extends HlsCachingProcessor {
public class CmafCachingProcessor implements Processor {

    // 子要素として動画とオーディオのm3u8を持つm3u8.
    // - 以下のgroup(1)とgroup(2)を合わせてキャッシュに必要な情報を通信毎に共有
    //   するためのキー(DomandCVI-Key, 連想配列のキーの意味)として使う.
    //   (searchに付いてくるsessionでもいいけど、path部の方が仕様変わりにくそうだから)
    // group(1): 24桁16進数. hlsbid. MongoDBのObjectIDと一致するらしい. たぶん毎回同じ.
    // group(2): 16桁16進数. m3u8の名前部分. 2024-06まではvideo mode(品質)ごとに一意の表現だった.
    private static final Pattern MASTER_PLAYLIST_URL_PATTERN = Pattern.compile(
        "^https?" + Pattern.quote("://delivery.domand.nicovideo.jp/") + "s?hlsbid/"
        + "([a-f0-9]+)/playlists/variants/([a-f0-9]+)" + Pattern.quote(".m3u8") + ".*");

    // masterの子であり、子要素としてvideo用key urlと無音動画チャンクurlリストを持つm3u8
    // - * dmc動画がdomandに移行し、"360p_low"(アンダースコアに続く"low")が現われるようになった場合
    //   * 、group(4)の正規表現を変更する必要がある. 今はその仕様は無いものとして扱う.
    //   * 2024-05追記. dmc動画はdmsへ移行した. "_low"は"-low"に移行したようだ. このコメントはもう
    //   * 消して良いかもしれない.
    // group(1): hlsbid.
    // group(2): video-src-id. 例: "video-h264-1080p", "video-h264-360p-lowest"
    // group(3): codec. 例: "h264",
    // group(4): 例: "1080p". "360p-lowest"
    // group(5): video height. "p"抜き. 例: "1080", "360"
    // group(6): いわゆるdmcLow. "" | "-lowest" | "-mid" | "-low"
    private static final Pattern VIDEO_PLAYLIST_URL_PATTERN = Pattern.compile(
        "https?" + Pattern.quote("://delivery.domand.nicovideo.jp/") + "s?hlsbid/"
        + "([a-f0-9]+)/playlists/media/(video-(\\w+)-((\\d+)p(-lowest|-mid|-low)?))" + Pattern.quote(".m3u8") + "[^\"]*"
        , Pattern.MULTILINE);

    // masterの子であり、子要素としてaudio用key urlと無音動画チャンクurlリストを持つm3u8
    // group(1): hlsbid.
    // group(2): audio-src-id. 例: "audio-aac-128kbps"
    // group(3): codec. 例: "aac",
    // group(4): audio kbps. "kbps"抜き. 例: "128"
    private static final Pattern AUDIO_PLAYLIST_URL_PATTERN = Pattern.compile(
        "https?" + Pattern.quote("://delivery.domand.nicovideo.jp/") + "s?hlsbid/"
        + "([a-f0-9]+)/playlists/media/(audio-(\\w+)-(\\d+)kbps)" + Pattern.quote(".m3u8") + "[^\"]*"
        , Pattern.MULTILINE);

    // group(1): "video" | "audio"
    // group(2): codec
    // group(3): video size or audio
    // group(4): "p" | "kbps"
    // group(5): "-lowest" | "-mid" | "-low" | ""
    private static final Pattern KEY_URL_PATTERN = Pattern.compile(
        "^https?" + Pattern.quote("://delivery.domand.nicovideo.jp/") + "s?hlsbid/"
        + "[a-f0-9]+/keys/(audio|video)-(\\w+)-(\\d+)(p|kbps)(-lowest|-mid|-low)?" + Pattern.quote(".key") + ".*");

    // group(1): "video" | "audio"
    // group(2): filename. 例: "init01.cmfv" | "02.cmfa"
    private static final Pattern CHUNK_URL_PATTERN = Pattern.compile(
        "^https?://(?:asset|delivery)" // sub-sub domain.
        + Pattern.quote(".domand.nicovideo.jp/") // domain.
        + "(?:s?hlsbid/)?" // (s)hlsbid magic.
        + "[a-f0-9]+/" // 16進数24桁.
        + "(?:segments/[a-f0-9]+/)?" // "shlsbid"の場合に入る.
        + "(audio|video)/" // group(1)
        + "\\d+/" // "1", "12", "123"...というバリエーションのある謎の数字.
        + "(?:(?:audio|video)-[^/]*)/" // 品質を示すっぽい表現.
        + "([^?]*).*" // group(2). ファイル名部分.
        );
    // 例: "https://delivery.domand.nicovideo.jp/shlsbid/1234567890abcdef12345678/segments/1234567890abcdef12345678/video/12/video-h264-720p/init001.cmfv?..."
    // 例: "https://asset.domand.nicovideo.jp/1234567890abcdef12345678/audio/1/audio-aac-64kbps/01.cmfa?..."

    // group(1): 動画ID. 例: "sm12345"
    private static final Pattern API_ACCESS_RIGHTS_HLS_URL_PATTERN = Pattern.compile(
        "^https?://" + Pattern.quote("nvapi.nicovideo.jp/v1/watch/")
        + "([^/?]*)" // group(1)
        + Pattern.quote("/access-rights/hls")
        + "(?:[?].*)?$"
        );
    // 例: https://nvapi.nicovideo.jp/v1/watch/sm1234/access-rights/hls?actionTrackId=1289abcxyz_1234567890123
    // - このURLの応答からマスタープレイリストが得られる.

    private final Executor executor;
    public final static ReentrantLock giantLock = new ReentrantLock();

    private static final String[] PROCESSOR_SUPPORTED_METHODS =
        new String[] { "GET", "POST" };
    private static final Pattern PROCESSOR_SUPPORTED_PATTERN = Pattern.compile(
        MASTER_PLAYLIST_URL_PATTERN.pattern()
        + "|" + VIDEO_PLAYLIST_URL_PATTERN.pattern()
        + "|" + AUDIO_PLAYLIST_URL_PATTERN.pattern()
        + "|" + KEY_URL_PATTERN.pattern()
        + "|" + CHUNK_URL_PATTERN.pattern()
        + "|" + API_ACCESS_RIGHTS_HLS_URL_PATTERN.pattern()
        );

    public CmafCachingProcessor(Executor executor) {
        this.executor = executor;
    }

    @Override
    public String[] getSupportedMethods() {
        return PROCESSOR_SUPPORTED_METHODS;
    }

    @Override
    public Pattern getSupportedURLAsPattern() {
        return PROCESSOR_SUPPORTED_PATTERN;
    }

    @Override
    public String getSupportedURLAsString() {
        return null;
    }

    static final Pattern URL_FOR_DEBUG = Pattern.compile("^(.*/)([^?/]*)([?].*)?$");
    public static String abbrurl(HttpRequestHeader requestHeader) {
        String rhash = String.format("%x", requestHeader.hashCode());
        String url = requestHeader.getURI();
        Matcher m = URL_FOR_DEBUG.matcher(url);
        if (m.matches()) {
            String o = m.group(1) + m.group(3);
            String name = m.group(2);
            // ファイル名部分/それ以外部分のハッシュ//:requestHeaderのハッシュ
            return String.format("%s/%x//:%s", name, o.hashCode(), rhash);
        };
        return rhash + "//" + url;
    };

    @Override
    public Resource onRequest(HttpRequestHeader requestHeader, Socket browser)
        throws IOException {

        // - ブラウザのリクエスト1回に対して複数回これが呼び出される前提で書く.
        // - なぜ複数来るかは不明. 分かった人教えてください.
        // - return null;は他のProcessorに処理を讓ることを意味する.

        // Logger.info("-- method: " + requestHeader.getMethod());

        String uri = requestHeader.getURI();

        Matcher m;
        if ((m = MASTER_PLAYLIST_URL_PATTERN.matcher(uri)).matches()) {
            // String hlsbid = m.group(1); // 24桁16進数
            // String m3u8id = m.group(2); // 16桁16進数
            return processMasterPlaylist(requestHeader);
        };

        if ((m = API_ACCESS_RIGHTS_HLS_URL_PATTERN.matcher(uri)).matches()) {
            return processApiHls(requestHeader, browser, m.group(1));
        };

        // - DomandCVIEntryは動画キャッシュ中に必要な情報を管理するコンテナ機能
        //   と、鍵とIVが揃った時まで処理を保留させるためのもの.
        DomandCVIEntry movieInfo = getDomandCVIEntry(requestHeader);
        if (movieInfo == null) {
            return null; // 高い確率でコーディングミスが原因.
        };

        {
            AV av = AV.UNSPECIFIED;
            Matcher mv = VIDEO_PLAYLIST_URL_PATTERN.matcher(uri);
            Matcher ma = AUDIO_PLAYLIST_URL_PATTERN.matcher(uri);
            if (mv.matches()) {
                av = AV.VIDEO;
            }
            else if (ma.matches()) {
                av = AV.AUDIO;
            };
            if (av != AV.UNSPECIFIED) {
                return processSubPlaylist(
                    requestHeader, av, movieInfo);
            };
        };

        if ((m = KEY_URL_PATTERN.matcher(uri)).matches()) {
            return processKey(requestHeader, /*"video"or"audio"*/m.group(1), movieInfo);
        };

        if ((m = CHUNK_URL_PATTERN.matcher(uri)).matches()) {
            return processChunk(
                requestHeader, movieInfo, /*"audio"or"video"*/m.group(1),
                /*filename*/m.group(2));
        };

        return null;
    };

    /**
     * - マスタープレイリストとsmidとの対応を保持する.
     * - 登録してすぐに使われるし、POSTメソッドno-cacheで要求されるため登録漏れする
     *   可能性も低い.
     * - そのため小さな数字でも問題が起きる可能性は低い.
     */
    static Map<String,String> masterPlaylistToSmid =
        Collections.synchronizedMap(new LRUMap<String,String>(50));

    private Resource processApiHls
    (HttpRequestHeader requestHeader, Socket browser, String smid) {
        // 例: https://nvapi.nicovideo.jp/v1/watch/sm1234/access-rights/hls?actionTrackId=1289abcxyz_1234567890123

        Pair<URLResource, byte[]> rr;
        try {
            rr = fetchBinaryContent(requestHeader, browser.getInputStream());
        } catch(IOException e) {
            Logger.info("nvapi通信エラー: " + smid);
            return null;
        };
        URLResource resource = rr.first;
        String content = new String(rr.second, StandardCharsets.UTF_8);

        // 本当はdareka.common.jsonを使ってjson解釈した方がいい.
        String t1 = "\"contentUrl\":\"";
        int p1 = content.indexOf(t1);
        if (p1 < 0) {
            // Logger.info("--ApiHls: p1 < 0: " + content);
            return resource;
        };
        int p2 = p1 + t1.length();
        int p3 = content.indexOf("\"", p2);
        if (p3 < 0) {
            // Logger.info("--ApiHls: p3 < 0: " + content);
            return resource;
        };
        String t4 = content.substring(p2, p3);
        int pq = t4.indexOf("?");
        if (pq >= 0) {
            t4 = t4.substring(0, pq);
        };

        masterPlaylistToSmid.put(t4, smid);
        // Logger.info("--mptosmid.put: " + smid + " , " + t4);

        return resource;
    };

    private Resource processKey(HttpRequestHeader requestHeader
                                , String videoOrAudio
                                , DomandCVIEntry movieInfo) {
        AV av = AV.UNSPECIFIED;
        if ("audio".equals(videoOrAudio)) {
            av = AV.AUDIO;
            // Logger.info("-- audio key: start");
        } else if ("video".equals(videoOrAudio)) {
            av = AV.VIDEO;
            // Logger.info("-- video key: start");
        } else {
            Logger.info("未知のkeyファイルです: " + movieInfo.getSmid());
            return null;
        };

        Pair<URLResource, byte[]> rr;
        try {
            rr = fetchBinaryContent(requestHeader);
        } catch(IOException e) {
            Logger.info("keyファイル通信エラー: " + movieInfo.getSmid());
            // movieInfo.setCacheSaveFlag(false)しない.
            return null;
        };
        URLResource resource = rr.first;
        byte[] binContent = rr.second;

        if (binContent.length != 16) {
            // 唯一対応しているAES-128の鍵長ではない
            Logger.info("AES-128 keyが16bytesではありません");
            // movieInfo.setCacheSaveFlag(false)しない.
            return resource;
        };

        // Logger.info("-- thread[" + Thread.currentThread().getId() + "]");

        // 既に鍵をセットしていても上書きする.
        if (av.isAudio()) {
            // Logger.info("-- audio key: ok");
            movieInfo.setAudioKey(binContent);
        } else {
            // Logger.info("-- video key: ok");
            movieInfo.setVideoKey(binContent);
        };

        return resource;
    };

    private static String getUrlBaseName(String url) {
        url = removeUrlSearch(url);
        int slash = url.lastIndexOf("/");
        if (0 <= slash) {
            url = url.substring(slash + 1);
        };
        return url;
    };

    private static String removeUrlSearch(String url) {
        int question = url.indexOf("?");
        if (0 <= question) {
            return url.substring(0, question);
        };
        return url;
    };

    // - nicocachenl_domandcvikeyからkeyを得て作成済みのDomandCVIEntryを得る.
    // - 作成済みのそれがない場合はnull.
    // - エラー表示処理.
    private static DomandCVIEntry getDomandCVIEntry(
        HttpRequestHeader requestHeader) {

        String key = requestHeader.getParameter("nicocachenl_domandcvikey");

        DomandCVIEntry data = null;
        if (key != null) {
            data = NLShared.INSTANCE.getDomandCVIManager().get(key);
        };
        if (key == null || data == null) {

            if ("true".equals(
                    requestHeader.getParameter("nicocachenl_noerror"))) {

                return null;
            };

            String k = key == null ? "ng" : "ok";
            String d = data == null ? "ng" : "ok";
            String videoType = requestHeader.getParameter("nicocachenl_video_type");
            String videoNumId = requestHeader.getParameter("nicocachenl_video_id");
            String noerror = requestHeader.getParameter("nicocachenl_noerror");
            String smid = "" + videoType + videoNumId;
            String m = "";
            if (videoType == null && videoNumId == null) {
                smid = "null"; // 若干見やすく.
            };
            if (key == null && data == null && !"null".equals(smid)) {
                // URL表示が鬱陶しいので、smidが得られた場合は下のelseifに比べて簡潔に表示する.
                Logger.info("CMAF付与情報取得失敗(1): (キー:" + k + ", 値:" + d
                            + ", smid:" + smid + ", noerror:" + noerror + ") "
                            + requestHeader.getPathBasename());
            } else if (key == null && data == null) {
                m = "(ページ更新で改善しない場合、おそらくインジェクション"
                    + "javascriptかnlFilterの構成が失敗しています) ";
                Logger.info("CMAF付与情報取得失敗(2): (キー:" + k + ", 値:" + d
                            + ", smid:" + smid + ", noerror:" + noerror + ") " + m
                            + requestHeader.getPathWithoutSearch());
            } else if (key != null && data == null) {
                Logger.info("ページを更新してください"
                            + "(キー:ok, 値:ng, smid:" + smid +")");
            };
            return null;
        };
        return data;
    };

    // - CMAFチャンクファイルの".cmfa"と"".cmfv"を入れておく場所.
    // - nltmp_smXXX*.hls ディレクトリがまだ存在しない場合はnull.
    public static File getTmpStreamCmafavDirectory(Cache cache, String audioOrVideo) {
        File tmpCacheDir = cache.getCacheTmpFile();
        if (tmpCacheDir == null || !tmpCacheDir.exists()) {
            return null;
        };
        return new File(tmpCacheDir, audioOrVideo);
    };

    private static VideoDescriptor superiorIncompatibleCache(DomandCVIEntry data) {
        String smid = data.getSmid();
        // swf,flv,mp4のキャッシュを持っている場合HLSをキャッシュしない
        if (Boolean.getBoolean("workaroundNoDisableDoubleCacheImported")) {
            return null;
        };
        VideoDescriptor cachedSmile = CacheManager.getPreferredCachedVideo(smid, false, null);
        if (cachedSmile != null && !cachedSmile.isLow()) {
            return cachedSmile;
        };
        VideoDescriptor cachedDmc = CacheManager.getPreferredCachedVideo(smid, true, null);
        if (cachedDmc != null && !cachedDmc.isLow()
            && !".hls".equals(cachedDmc.getPostfix())) {

            return cachedDmc;
        };
        return null;
    };

    private static int notifyAndCheckResult(NLEventSource eventSource, int eventId) {
        if (eventSource != null) {
            int result = NLShared.INSTANCE.notifySystemEvent(eventId, eventSource, true);
            if (result != SystemEventListener.RESULT_OK) {
                return result;
            }
        }
        return SystemEventListener.RESULT_OK;
    }

    private static final Pattern NUMBER_PATTERN = Pattern.compile("[0-9]");
    private static Pair<String,String> getVideoTypeAndId
    (HttpRequestHeader requestHeader) {

        // - nicocachenl_video_typeとnicocachenl_video_idは
        //   url_injection_sys.jsというブラウザ側jsが付加していたが、これは廃止した
        //   (2024-10-04).

        // 例:"sm".
        String type = requestHeader.getParameter("nicocachenl_video_type");

        // 例:sm9なら"9".
        String id = requestHeader.getParameter("nicocachenl_video_id");

        if (type != null && id != null) {
            return new Pair<String,String>(type, id);
        };

        String url = requestHeader.getURI();
        int question = url.indexOf("?");
        if (question >= 0) {
            url = url.substring(0, question);
        };

        // Logger.info("--mptosmid.get: " + url);
        String smid = masterPlaylistToSmid.get(url);
        if (smid == null) {
            return new Pair<String,String>(null, null);
        };

        Matcher m = NUMBER_PATTERN.matcher(smid);
        if (m.find()) {
            return new Pair<String,String>(
                smid.substring(0, m.start()),
                smid.substring(m.start()));
        };
        return new Pair<String,String>(null, null);
    };

    private Resource processMasterPlaylist
    (HttpRequestHeader requestHeader)
        throws IOException {

        // Logger.info(abbrurl(requestHeader));
        // Logger.info(requestHeader.toString());

        String uri = requestHeader.getURI();

        Pair<String,String> videoTypeAndId = getVideoTypeAndId(requestHeader);
        String videoType = videoTypeAndId.first;
        String videoId = videoTypeAndId.second;
        videoTypeAndId = null;
        String smid = videoType + videoId;

        if (videoType == null || videoId == null) {
            // nicocache_nlのjavascriptによるインジェクションが上手くいっていない
            Logger.info("対象URL(cmaf)ですが動画情報が不明なためキャッシュしません: " + smid);
            Logger.debug("url: " + uri);
            return null;
        };

        // Logger.info("---- 286cmaf " + videoType + " " + videoId + " " + uri);

        // video_src_idとaudio_src_idを得るためにbodyを得る.
        Pair<URLResource, byte[]> rr = fetchBinaryContent(requestHeader);
        URLResource resource = rr.first;
        byte[] binContent = rr.second;

        if (binContent == null) {
            return resource;
        };

        String masterM3u8;
        masterM3u8 = new String(binContent, StandardCharsets.UTF_8);

        // - ここからmaster.m3u8に書かれた内容をパースしている.
        // - 正規表現を使った無理矢理な方法.
        // - 正当にはm3u8パーサーを用意すること.

        Matcher videoPLM = VIDEO_PLAYLIST_URL_PATTERN.matcher(masterM3u8);
        Matcher audioPLM = AUDIO_PLAYLIST_URL_PATTERN.matcher(masterM3u8);

        DomandCVIEntry audioMovInfo = null;

        // エラー用. もしaudioのsub-playlistが見つからない場合は0のまま.
        int subPlaylistCount = 0;

        if (audioPLM.find()) {
            String audioSrcId = audioPLM.group(2); // 例: "audio-aac-128kbps"
            String audioKbps = audioPLM.group(4); // 例: 128
            String keyForAudioDomandCVI = smid + audioSrcId;

            // - nltmp_sm12345[0p,128]_title.hls という一時dirを作る.
            // - そこにaudio chunkだけをキャッシュする.
            // - 完了時にvideoキャッシュ側に統合する.
            // - このコードは複数のaudioを想定していない.
            audioMovInfo = prepareDomandCVIEntry(
                keyForAudioDomandCVI, videoType, videoId, "0", audioKbps,
                "0p", null, audioSrcId);

            if (audioMovInfo == null) {
                // - prepareDomandCVIEntryがログを済ましている.
                // - nicocachenl_noerrorを伝播させる処理をするべき.
                return resource;
            };

            // - 複数のvideoプレイリストがある.
            // - 1080p, 720p, 480p, 320p, 320p-lowestなどそれぞれにDomandCVIEntryを
            //   用意する.
            while (videoPLM.find()) {
                ++subPlaylistCount;

                String videoSrcId = videoPLM.group(2); // 例: "video-h264-1080p"
                String videoHeight = videoPLM.group(5); // 例: 1080
                String videoMode = videoPLM.group(4); // 例: 1080p
                String keyForVideoDomandCVI = smid + videoSrcId;
                DomandCVIEntry videoMovInfo = prepareDomandCVIEntry(
                    keyForVideoDomandCVI,
                    videoType, videoId, videoHeight, audioKbps,
                    videoMode, videoSrcId, audioSrcId);
                // nullはどういう状況か？
                if (videoMovInfo != null) {
                    videoMovInfo.assocList.add(audioMovInfo);
                    audioMovInfo.assocList.add(videoMovInfo);
                }
            };
        };

        if (0 == subPlaylistCount) {
            Logger.info("サブプレイリストを検出出来ないためキャッシュしません: " + smid);
            return resource;
        };

        // - 利用できるうちの最高画質.
        // - 1080pで投稿された動画であっても一般会員は720p(2024-08時点の制限).
        DomandCVIEntry topVideoMovInfo = audioMovInfo.assocList.get(0);

        // - 一番画質が良いものでイベント通知する.
        // - 全ての画質を通知するべきか？期待される動作が分からない.
        Cache cacheForEvent = topVideoMovInfo.getCache();

        NLEventSource eventSource = null;
        if (NLShared.INSTANCE.countSystemEventListeners() > 0) {
            eventSource = new NLEventSource(null, requestHeader, cacheForEvent);
        };

        // [nl] Extensionにキャッシュ要求イベントを通知する.
        if   (notifyAndCheckResult(eventSource, SystemEventListener.CACHE_REQUEST)
              != SystemEventListener.RESULT_OK) {
            audioMovInfo.setCacheSaveFlag(false); // video側にも伝播する.
            Logger.debug(requestHeader.getURI() + " pass-through by extension");
            return resource;
        };

        // - 特殊キャッシュを持っていたらキャッシュから応答する.
        Resource cacheResource = processMasterPlaylistFromCacheIfExists(
            requestHeader, topVideoMovInfo);
        if (cacheResource != null) {
            return cacheResource;
        };

        // [nl] Extensionがキャッシュを禁止する場合はキャッシュしない
        if   (notifyAndCheckResult(eventSource, SystemEventListener.CACHE_STARTING)
              != SystemEventListener.RESULT_OK) {
            audioMovInfo.setCacheSaveFlag(false); // video側にも伝播する.
            Logger.info("(cmaf|ext)disable cache: " +
                        topVideoMovInfo.getCache().getCacheFileName());
            // fall through
        };

        // smidにtitleを結び付ける処理.
        scheduleTitleRetrieverIfNeeded(audioMovInfo);

        // キャッシュしない場合でも、そのフラグを子要素であるURLに伝達するために通常通り
        // のことをする.
        return processMasterPlaylistFromServer(
            requestHeader, audioMovInfo, resource, masterM3u8);
    };

    // - メソッド名は「キャッシュから応答するわけではない」という意味.
    private Resource processMasterPlaylistFromServer
    (HttpRequestHeader requestHeader
     , DomandCVIEntry audioMovInfo
     , URLResource resource, String masterM3u8) throws IOException {

        // Logger.info("----(cmaf)processMasterPlaylistFromServer: " + requestHeader.getPathBasename());

        {
            String contentToWriteFile = buildMasterM3u8ForSaving(masterM3u8);
            if (contentToWriteFile != null) {
                // movieInfo.chunkLoadStart()した時に実際に書き込まれる.
                byte[] bytes = contentToWriteFile.getBytes(StandardCharsets.UTF_8);
                audioMovInfo.mightWriteMasterM3u8(bytes);
                for (DomandCVIEntry videoMovInfo : audioMovInfo.assocList) {
                    videoMovInfo.mightWriteMasterM3u8(bytes);
                };
            } else {
                audioMovInfo.setCacheSaveFlag(false); // video側にも伝播する.
                Logger.info("(cmaf)MasterPlaylistの加工に失敗しました。プログラミングエラーです。");
            };
        };
        // キャッシュ保存処理終わり.

        // - 通信ハンドラーにキャッシュに必要な情報を渡すために、サーバーから来た
        //   マスタープレイリストを加工してからクライアントへ.

        boolean[] firstAudio = {true}; // lambda内で変更するためだけに配列として宣言.
        String contentToResponse = M3u8Util.replaceURL(masterM3u8, (url) -> {
                if (url.contains("/audio") && firstAudio[0]) {
                    firstAudio[0] = false;
                    return addUrlSearch(
                        url,
                        "nicocachenl_domandcvikey=" + audioMovInfo.getKey());
                };

                if (!url.contains("/video")) {
                    Logger.info("error: " + audioMovInfo.getSmid()
                                + ": unknown url expression: " + url);
                    return addUrlSearch(url, "nicocachenl_noerror=true");
                };

                String videoSrcId = getVideoSrcId(url); // 例: "video-h264-1080p"
                if (videoSrcId == null) {
                    Logger.info("error: " + audioMovInfo.getSmid()
                                + ": unknown video m3u8 url expression: " + url);
                    return addUrlSearch(url, "nicocachenl_noerror=true");
                };
                DomandCVIEntry videoMovInfo =
                    audioMovInfo.getDomandCVIEntryByVideoSrcId(videoSrcId);
                if (videoMovInfo == null) {
                    Logger.info("error: " + audioMovInfo.getSmid()
                                + ": unknown video-src-id: " + videoSrcId);
                    return addUrlSearch(url, "nicocachenl_noerror=true");
                };
                return addUrlSearch(
                    url,
                    "nicocachenl_domandcvikey=" + videoMovInfo.getKey());
            });

        Resource resp = new StringResource(contentToResponse);
        resp.setResponseHeader(HttpHeader.CONTENT_TYPE, "application/vnd.apple.mpegurl");
        setCors(resp, requestHeader);
        // LimitFlvSpeedListener.addTo(resp);
        return resp;
    };

    private static String addUrlSearch(String url, String search) {
        if (url.contains("?")) {
            return url + "&" + search;
        };
        return url + "?" + search;
    };

    private static String getVideoSrcId(String m3u8url) {
        Matcher videoPLM = VIDEO_PLAYLIST_URL_PATTERN.matcher(m3u8url);
        if (!videoPLM.find()) {
            return null;
        };
        return videoPLM.group(2); // 例: "video-h264-1080p"
    };

    // ロギング用.
    private static DomandCVIEntry prepareDomandCVIEntry
    (String entryKeyName
     , String videoType, String videoNumId, String videoHeight
     , String audioKbps , String videoMode, String videoSrcId
     , String audioSrcId) {

        DomandCVIEntry movieInfo
            = NLShared.INSTANCE.getDomandCVIManager().get(entryKeyName);

        if (movieInfo != null) {
            movieInfo.restart();
            return movieInfo;
        };
        try {
            // - ここがDomandCVIEntryの初期化ポイント.
            // - DomandCVIManagerにも入る. keyは連想配列のキー.
            movieInfo = DomandCVIUtil.initAndPutEntry(
                /*key*/entryKeyName, videoType, videoNumId, videoHeight
                , audioKbps, videoMode, videoSrcId, audioSrcId);
        } catch (NoIdInfoException e) {
            // どういう場合か不明. 要検証.
            Logger.info("idInfo is not found: " + videoType + videoNumId);
            return null;
        } catch (NumberFormatException e) {
            Logger.info("video height or audio kbps is not integer expression: "
                        + videoType + videoNumId);
            return null;
        };
        return movieInfo;
    };

    private static String buildMasterM3u8ForSaving(String content) {
        // - キャッシュ保存用master.m3u8内容を作る. 失敗時はnull.
        // - contentは公式サーバーから送られてくるマスターm3u8内容.
        // - マスターm3u8内容を元に作る必然性はない.
        //   仕様変更に自動的に追従しそうな気配を理由にこの実装を選ぶ.

        // {video,audio}プレイリストをキャッシュ用表現にするためにreplace.
        // - processMasterPlaylistでMatch済み. 2回目で格好悪い.
        // - コード密度が高くて読み辛い.
        Matcher videoPLM = VIDEO_PLAYLIST_URL_PATTERN.matcher(content);
        String content01 = content;
        if (videoPLM.find()) {
            // - 【縮退実装2024-08-08】
            // - 一番上に来るvideo用m3u8が最も品質が良いことを前提にした処理.
            // - それ以降は記録する必要はないから削除のためにsubstring.
            content01 = content.substring(0, videoPLM.end()) + "\n";
            videoPLM = VIDEO_PLAYLIST_URL_PATTERN.matcher(content01);
        };
        String contentV = videoPLM.replaceFirst("video.m3u8");
        boolean notReplaced = content.equals(contentV);
        Matcher audioPLM = AUDIO_PLAYLIST_URL_PATTERN.matcher(contentV);
        String contentVA = audioPLM.replaceFirst("audio.m3u8");
        notReplaced = notReplaced || contentV.equals(contentVA);
        if (notReplaced) {
            return null;
        };
        return contentVA;
    };


    public static int countString(String haystack, String needle) {
        int needleLength = needle.length();
        int count = 0;
        int index = 0;

        for (;;) {
            index = haystack.indexOf(needle, index);
            if (index < 0) {
                return count;
            };
            ++count;
            index += needleLength;
        }
    };

    private void scheduleTitleRetrieverIfNeeded(DomandCVIEntry data) {
        FutureTask<String> retrieveTitlteTask = null;
        if (  Boolean.getBoolean("title")
              && (data.getIdInfo() == null || !data.getIdInfo().isTitleValid())) {
            retrieveTitlteTask = new FutureTask<>(
                new NicoCachingTitleRetriever(
                    data.getVideoType(), data.getVideoNumId()));
            executor.execute(retrieveTitlteTask);
        };
    };

    // - キャッシュを持っていればローカルのaudio.m3u8とvideo.m3u8を持つマスタープレイリストを返す.
    // - 持っていなければnull.
    private Resource processMasterPlaylistFromCacheIfExists
    (HttpRequestHeader requestHeader, DomandCVIEntry data)
        throws IOException {

        String smid = data.getSmid();
        VideoDescriptor video = data.getVideoDescriptor();
        String postfix = data.getPostfix();

        // - 持っている中での最高品質を返す.
        // - 要求に合わせた品質キャッシュを応答する機能は実装していない.
        VideoDescriptor cachedHls =
            CacheManager.getPreferredCachedVideo(smid, true, Cache.HLS);

        Logger.debug("Preferred cache: " + cachedHls);
        if (cachedHls == null || video.isPreferredThan(cachedHls, true, postfix)) {
            return null;
        };

        Cache cache = new Cache(cachedHls);
        if (!cache.exists()) {
            // nltmpを持っている場合もここを通る.
            return null;
        };
        data.setCache(cache);

        File cacheDir = cache.getCacheFile();
        if (new File(cacheDir, "video.m3u8").exists()
            && new File(cacheDir, "audio.m3u8").exists()) {

            return null; // dms仕様の普通のhlsキャッシュ.
        };
        if (new File(cacheDir, "1/ts/playlist.m3u8").exists()) {
            return null; // dmc/hls仕様の普通のhlsキャッシュ.
        };
        // - 上記の除外条件ではprocessSubPlaylistFromCacheIfExistsでキャッシュ利用処理する.
        // - 上記以外のmaster.m3u8だけを持っているだろう未知のhlsキャッシュ.
        // - マスタープレイリストに書かれたcodecと実際のサブプレイリストのチャンクコーデックが
        //   不一致だと再生されない.
        // - ここではマスタープレイリストを乗っ取ることでその問題を回避する.
        // - 下記方法でマスターを乗っ取ると、画質選択の表示がおかしくなる(2024-08).
        //   - 最低画質を選択しているように表示されたり、自動を選択しているように表示される.
        //   - これらの画質選択は保存されるわけではないから実害はない.
        //   - しかし格好悪いので、上記条件により除外し最低限にする.
        //   - 本当は品質ごとのプレイリストが存在しているように偽装するべき.

        File file = new File(cacheDir, "master.m3u8");

        boolean loadFailed = false;
        String urlPrefix = requestHeader.getScheme() + "://" + requestHeader.getHost();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (BufferedReader br = new BufferedReader(new FileReader(file))) {
            String params = "nicocachenl_domandcvikey=" + data.getKey();
            String line;
            while (null != (line = br.readLine())) {
                // - CmafUseCacheProcessorを呼び出すURLをセット.
                line = M3u8Util.replaceURL(line, url -> {
                        return urlPrefix + "/cache/file/" + params + "//" + url;
                    });
                baos.write(line.getBytes(StandardCharsets.UTF_8));
                baos.write('\n');
            };
        } catch (IOException e) {
            loadFailed = true;
        };

        if (loadFailed) {
            Logger.warning("(cmaf|master)load failed: master.m3u8: " + file.getPath());
        } else {
            Logger.info("(cmaf)using cache: " + cache.getCacheFileName());
            if (Boolean.getBoolean("touchCache")) {
                cache.touch();
            };
        };

        Resource r = new StringResource(baos.toByteArray());
        r.setResponseHeader(HttpHeader.CONTENT_TYPE, "application/vnd.apple.mpegurl");
        setCors(r, requestHeader);

        // LimitFlvSpeedListener.addTo(r);
        return r;
    };


    private static Resource processSubPlaylistFromCacheIfExists
    (HttpRequestHeader requestHeader, AV av, DomandCVIEntry movieInfo) {

        // - movieInfo.cacheはこのメソッドで初期化(newされ代入)される.
        //   キャッシュ利用せずキャッシュ保存する処理でもmovieInfo.cacheは利用される.

        String smid = movieInfo.getSmid();
        VideoDescriptor video = movieInfo.getVideoDescriptor();
        String postfix = movieInfo.getPostfix(); // ".hls"

        // - 持っている中での最高品質を返す.
        // - 要求に合わせた品質キャッシュを応答する機能は実装していない.
        Cache cache = movieInfo.getCache();
        if (cache == null || !cache.exists()) {
            VideoDescriptor cachedHls =
                CacheManager.getPreferredCachedVideo(smid, true, Cache.HLS);

            Logger.debug("Preferred cache: " + cachedHls);
            if (cachedHls == null || video.isPreferredThan(cachedHls, true, postfix)) {
                return null;
            };

            cache = new Cache(cachedHls);
            movieInfo.setCache(cache);
        };

        if (Boolean.getBoolean("touchCache")) {
            cache.touch();
        };

        File cacheDir = cache.getCacheFile();
        File file = null;
        if (av.isAudio()) {
            file = new File(cacheDir, "audio.m3u8");
        } else {
            file = new File(cacheDir, "video.m3u8");
        };
        if (!file.exists()) {
            // - hlsキャッシュだがaudio.m3u8かvideo.m3u8が存在しない.
            // - 2024-08-20時点でニコニコ動画のhlsプレイヤーはaudio.m3u8側とvideo.m3u8側に、
            //   master.m3u8内容を返しても期待通りの再生をする.
            // - audio側にもvideo側にも同じ動画チャンクが渡る. これは意図通り.
            // - dmc時代のhlsキャッシュ.
            //   (smXXX*.hls/下にmaster.m3u8, 1/ts/playlist.m3u8, 1/ts/1.tsなどを持つ)
            // - あるいは過去のNicoCacheが作ったのではない独自のhlsキャッシュ.
            file = new File(cacheDir, "master.m3u8");
        };

        boolean loadFailedm3u8 = false;
        String urlPrefix = requestHeader.getScheme() + "://" + requestHeader.getHost();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (BufferedReader br = new BufferedReader(new FileReader(file))) {
            String params = "nicocachenl_domandcvikey=" + movieInfo.getKey();
            String line;
            // - なぜ一行ずつ処理することにしたか覚えていない.
            while (null != (line = br.readLine())) {
                // - CmafUseCacheProcessorを呼び出すNicoCache URLをセット.
                line = M3u8Util.replaceURL(line, url -> {
                        return urlPrefix + "/cache/file/" + params + "//" + url;
                    });
                baos.write(line.getBytes(StandardCharsets.UTF_8));
                baos.write('\n');
            };
        } catch (IOException e) {
            loadFailedm3u8 = true;
        };

        String reqbasename = getCodecAndRate(getUrlBaseName(requestHeader.getURI()));
        if (loadFailedm3u8) {
            Logger.warning("(cmaf|sub)load failed: " + file.getPath());
        } else if (av.isAudio()) {
            Logger.info("(cmaf)using audio cache: " + smid
                        + ": req: " + reqbasename);
        } else {
            Logger.info("(cmaf)using video cache: " + smid
                        + ": req: " + reqbasename);
        };

        Resource r = new StringResource(baos.toByteArray());
        r.setResponseHeader(HttpHeader.CONTENT_TYPE, "application/vnd.apple.mpegurl");
        setCors(r, requestHeader);

        // LimitFlvSpeedListener.addTo(r);
        return r;
    };

    // - 表示のためのメソッド.
    // - "video-h264-1080p.m3u8" -> "h264-1080p"
    // - "audio-aac-192kbps.m3u8" -> "aac-192kbps"
    private static String getCodecAndRate(String s) {
        int start = s.indexOf('-');
        int end = s.lastIndexOf('.');
        if (start < 0 || end < 0) {
            return s;
        };
        return s.substring(start + 1, end);
    };

    private static final Pattern EXT_X_KEY_LINE =
        Pattern.compile("(^|\n)(#EXT-X-KEY:[^\n]*?)(,[^\n]*)?(\n|$)");

    // cmaf-m3u8の#EXT-X-KEY:行をただのコメント行にする.
    private static String removeExtXKeyLine(String playlistContent) {
        Matcher m = EXT_X_KEY_LINE.matcher(playlistContent);
        return m.replaceAll("$1#$4"); // 行数減らさない.
    };

    // group(1): "audio" | "video"
    // group(2): 品質ごとに最高品質から低品質へ向かって"1","12","123"と増えていく謎の部分
    // group(3): チャンクファイル名
    private static final Pattern TO_SUBPLAYLIST_REMOTE_URL_TO_LOCAL_URL =
            Pattern.compile("^.*/(audio|video)/(\\d+)/[^/]*/([^?]*)(?:[?].*)$");

    // group(1): method. 例: AES-128
    // group(2): key url.
    // group(3): Initialization Vector. 例: 0x1E1C49FE49E2BA5AC9CBA6BB0CF74B4C
    private static final Pattern EXT_X_METHOD_AND_KEY_URL_AND_IV_PATTERN =
        Pattern.compile("(?:^|\n)#EXT-X-KEY[^\n]*METHOD=([^,\n]*)[^\n]*URI=\"([^\"]*)[^\n]*IV=([^\n,]*)");


    private static Resource processSubPlaylist(
        HttpRequestHeader requestHeader, AV av, DomandCVIEntry movieInfo)
        throws IOException {

        // Logger.info("---- cmaf processSubPlaylist " + hash);

        if (av.isUnspecified()) {
            Logger.info("processSubPlaylist: av==UNSPECIFIED: コーディングミス");
            return null;
        };
        if (movieInfo == null) {
            Logger.info("processSubPlaylist: movieInfo==null: コーディングミス");
            return null;
        };

        Resource preferredCache = processSubPlaylistFromCacheIfExists(
            requestHeader, av, movieInfo);
        if (preferredCache != null) {
            return preferredCache;
        };

        // swf,flv,mp4のキャッシュを持っている場合HLSをキャッシュしない
        {
            VideoDescriptor cached = superiorIncompatibleCache(movieInfo);
            if (cached != null) {
                movieInfo.setCacheSaveFlag(false); // キャッシュしないというフラグ.
                Logger.info("(cmaf|sub)single file cache found. disable cache: "
                            + cached.toString());
                // no return.
            };
        };

        // - キャッシュしない決定をprocessMasterPlaylistでしていても、URLの加工は行う.

        Pair<URLResource, byte[]> rr = fetchBinaryContent(requestHeader);
        URLResource serverResponse = rr.first;
        byte[] binContent = rr.second;

        if (binContent == null) {
            Logger.info("(CCP|binContent)error: 通信失敗.");
            return serverResponse;
        };

        String content = new String(binContent, StandardCharsets.UTF_8);

        // - 余計なsearch部を付けた場合や大量にリクエストを送った時に起きる.
        // - dms時代は403なし. dmc時代は403が付いていた. 一応403もチェック.
        if (content.startsWith("Forbidden") ||
            content.startsWith("403 Forbidden")) {

            movieInfo.setCacheSaveFlag(false); // キャッシュしない
            Logger.info("(cmaf subpl)Forbidden: requestHeader:" + requestHeader);
        };

        if (movieInfo.getCacheSaveFlag()) {
            String contentToWriteFile = SubPlaylistRemoteUrlToLocalCacheUrl(
                content, movieInfo);
            // contentToWriteFile==nullならばsetCacheSaveFlag(false)済み.

            if (contentToWriteFile != null) {
                byte[] binContentToWriteFile = contentToWriteFile.getBytes(StandardCharsets.UTF_8);
                if (av.isAudio()) {
                    movieInfo.mightWriteAudioM3u8(binContentToWriteFile);
                } else {
                    movieInfo.mightWriteVideoM3u8(binContentToWriteFile);
                };
            };
        };

        String encIVHex; // 復号情報の初期化ベクトルの文字列形式. "0x789ABC..."
        {
            Matcher m = EXT_X_METHOD_AND_KEY_URL_AND_IV_PATTERN.matcher(content);
            if (!m.find()) {
                movieInfo.setCacheSaveFlag(false); // キャッシュしないというフラグ.
                Logger.info("復号情報をプレイリストから取り出せませんでした: "
                            + movieInfo.getSmid());
                return serverResponse;
            };
            String encMethod = m.group(1);
            String encKeyUrl = m.group(2);

            if (KEY_URL_PATTERN.matcher(encKeyUrl).matches()) {
                // Logger.info("-- " + av + " key url: 形式ok");
                // do nothing.
            }
            else {
                Logger.info("" + av + " key url: 未知形式 '" + encKeyUrl + "'");
            };

            encIVHex = m.group(3);

            if (!("AES-128".equals(encMethod))) {
                movieInfo.setCacheSaveFlag(false); // キャッシュしないというフラグ.
                Logger.info("非対応の暗号化方法(" + encMethod + ")です: "
                            + movieInfo.getSmid());
            };
        };

        if (!encIVHex.startsWith("0x")) {
            movieInfo.setCacheSaveFlag(false); // キャッシュしないというフラグ.
            Logger.info("非対応の初期化ベクトル表現です: "
                        + movieInfo.getSmid());
        };

        byte[] encIV; // 復号情報の初期化ベクトル. AES-128では16bytes.
        try {
            encIV= hexStringToByteArray(encIVHex.substring(2));
        } catch (NumberFormatException e) {
            movieInfo.setCacheSaveFlag(false); // キャッシュしないというフラグ.
            Logger.info("初期化ベクトル表現が不正です:["
                        + encIVHex + "]: " + movieInfo.getSmid());
            return serverResponse;
        };

        if (av.isAudio()) {
            movieInfo.setAudioIV(encIV);
        } else {
            movieInfo.setVideoIV(encIV);
        };
        // Logger.info("-- " + av + " iv: ok");

        // メソッドが縦に長すぎる. 整理必要.

        // - レスポンスを加工してクライアント(ブラウザ)へ.
        // - ここには未知形式のURLを含むm3u8は来ない. 未知形式は、
        //   SubPlaylistRemoteUrlToLocalCacheUrlで検証されエラー済み.
        String contentToResponse = M3u8Util.injectURLSearch(
            content, "nicocachenl_domandcvikey=" + movieInfo.getKey());
        Resource resp = new StringResource(contentToResponse);
        resp.setResponseHeader(HttpHeader.CONTENT_TYPE, "application/vnd.apple.mpegurl");
        setCors(resp, requestHeader);
        // LimitFlvSpeedListener.addTo(resp);
        return resp;
    };

    // - SubPlaylist(例:video-h264-1080p.m3u8,audio-aac-128kbps.m3u8)の
    //   ニコ動側のm3u8内容をローカルキャッシュ用に変換する.
    // - 不正な表現があった場合にキャッシュしないフラグをセット.
    // - エラーの表示.
    private static String SubPlaylistRemoteUrlToLocalCacheUrl
    (String remoteContent, DomandCVIEntry movieInfo) {

        String contentToWriteFile = removeExtXKeyLine(remoteContent);
        try {
            contentToWriteFile = M3u8Util.replaceURL(
                contentToWriteFile,
                (url) -> {
                    if (!PROCESSOR_SUPPORTED_PATTERN.matcher(url).matches()) {
                        // - CmafCachingProcessorが処理しない表現が表われること
                        //   は想定外. 他でエラーするからここで弾く.
                        throw new IllegalStateException(url);
                    };
                    Matcher m = TO_SUBPLAYLIST_REMOTE_URL_TO_LOCAL_URL.matcher(url);
                    if (!m.find()) {
                        throw new IllegalStateException(url);
                    };
                    String audioVideo = m.group(1);
                    String filename = m.group(3);
                    return m.replaceFirst(audioVideo + "/" + filename);
                });
        } catch (IllegalStateException e) {
            movieInfo.setCacheSaveFlag(false);
            Logger.info("サブプレイリストに不明な表現があるためキャッシュしません"
                        + ": '" + e.getMessage() + "': " + movieInfo.getSmid());
            return null;
        };
        return contentToWriteFile;
    };

    private static Pair<URLResource, byte[]> fetchBinaryContent(
        HttpRequestHeader requestHeader) throws IOException {

        return fetchBinaryContent(requestHeader, null);
    };

    private static Pair<URLResource, byte[]> fetchBinaryContent
    (HttpRequestHeader requestHeader, InputStream browserToServer)
        throws IOException {

        String uri = requestHeader.getURI();
        requestHeader.removeHopByHopHeaders();

        // 解凍できないEncodingは削除
        String acceptEncoding = requestHeader.getMessageHeader(HttpHeader.ACCEPT_ENCODING);
        if (acceptEncoding != null) {
            // Logger.info("--acceptEncoding ori: " + acceptEncoding);
            acceptEncoding = acceptEncoding.toLowerCase().replaceAll(
                "(?: *, *)?(?:bzip2|sdch|br)", "");
            acceptEncoding = acceptEncoding.replaceFirst("^ *, *", "");
            // Logger.info("--acceptEncoding rep: " + acceptEncoding);
            requestHeader.setMessageHeader(HttpHeader.ACCEPT_ENCODING, acceptEncoding);
        }

        // ヘッダを受信して、Bodyも受信するか判断
        URLResource r = new URLResource(uri);
        HttpResponseHeader responseHeader
            = r.getResponseHeader(browserToServer, requestHeader);
        if (responseHeader == null) {
            Logger.warning("failed to access to: " + uri + " (no responseHeader)");
            return new Pair<>(r, null);
        }
        responseHeader.removeHopByHopHeaders();

        responseHeader.removeMessageHeader("Vary");
        responseHeader.removeMessageHeader("Accept-Ranges");

        // Bodyの取得
        byte[] bcontent = r.getResponseBody();
        if (bcontent == null) {
            Logger.warning("failed to access to: " + uri + " (no responseBody)");
            return new Pair<>(r, null);
        }

        return new Pair<>(r, bcontent);
    }

    private Resource processChunk(HttpRequestHeader requestHeader
                                  , DomandCVIEntry movieInfo
                                  , String avword
                                  , String filename)
        throws IOException {

        if (movieInfo == null) {
            Logger.info("processChunk: movieInfo==null: コーディングミス");
            return null;
        };

        if (!movieInfo.getCacheSaveFlag()) {
            return null;
        };

        AV av = AV.UNSPECIFIED;
        if ("audio".equals(avword)) {
            av = AV.AUDIO;
        } else if ("video".equals(avword)) {
            av = AV.VIDEO;
        };

        if (av.isUnspecified()) {
            Logger.info("引数が\"audio\"でも\"video\"でもありません[" + avword + "]");
            return null;
        };

        NLEventSource eventSource = null;
        if (NLShared.INSTANCE.countSystemEventListeners() > 0) {
            eventSource = new NLEventSource(null, requestHeader,
                                            movieInfo.getCache());
        };

        // - チャンク読み込みの度に呼び出す.
        // - 関連付けられた処理は一度のみ走る.
        // - これをトリガーにnltmp_smXXX*.hlsが作られm3u8が保存される.
        movieInfo.chunkLoadStart();

        // TODO: 判定がとても雑. しかし正当に対応するにはm3u8のパースをして、
        //       ファイル名と属性の管理をしなければならない.
        // - init以降には数値1. 左0パディングは動画チャンク数の桁数で決まる.
        //   例: init1.cmfa, init01.cmfv, init001.cmfv
        // - trueになる想定filenameは01.cmfv, 999.cmfv, 01.cmfa, 123.cmfaなど.
        boolean needDecrypt = !(filename.startsWith("init"));

        URLResource serverResource;
        try {
            Cache.incrementDL(movieInfo.getVideoDescriptor());

            String url = requestHeader.getURI();
            serverResource = new URLResource(url);
            serverResource.setFollowRedirects(true);

            ChunkListener x = new ChunkListener(
                movieInfo, av, filename, needDecrypt, eventSource, executor);
            if (url.contains("/shlsbid/")) {
                // - shlsbid動画は仕様が違う.
                // - 一部のログを非表示に.
                x.setShlsbid(true);
            };

            serverResource.addTransferListener(x);
        } catch (RuntimeException e) {
            Logger.error(e);
            throw e;
        };
        return serverResource;
    };


    private static void setCors(Resource r, HttpRequestHeader requestHeader) {
        if (requestHeader.getMessageHeader("Origin") != null) {
            r.setResponseHeader("Access-Control-Allow-Credentials", "true");
            r.setResponseHeader("Access-Control-Allow-Origin", requestHeader.getMessageHeader("Origin"));
        }
    }

    // "0x"で始まらない16進数文字列をバイト配列へ.
    // TODO: utility関数は適切な場所へ移動すること.
    // based on https://stackoverflow.com/questions/140131/
    //          author: https://stackoverflow.com/users/3093/dave-l
    public static byte[] hexStringToByteArray(String s)
        throws NumberFormatException {

        int len = s.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            int x1 = (byte)Character.digit(s.charAt(i), 16);
            int x2 = (byte)Character.digit(s.charAt(i+1), 16);
            if (x1 == -1 || x2 == -1) {
                throw new NumberFormatException();
            };
            data[i / 2] = (byte)((x1 << 4) + x2);
        };
        return data;
    };
};


// - Runnableを指定のexecutorでexecuteするようにするためのラッパークラス.
// - 復号情報取得時に別スレで復号を開始させるために定義.
// - 同期的に復号すると鍵要求の応答が遅れるから.
// - 動詞的なクラス名ってどうなの.
class AsyncExecute implements Runnable {
    Runnable runnable;
    Executor executor;
    public AsyncExecute(Runnable runnable, Executor executor) {
        this.runnable = runnable;
        this.executor = executor;
    };

    @Override
    public void run() {
        executor.execute(runnable);
    };
};


class ChunkListener implements TransferListener, Runnable {
    Cache cache;
    DomandCVIEntry movieInfo;
    NLEventSource eventSource;
    Executor executor;

    // - ローカル側セグメントファイル名.
    // - 個別キャッシュ(smXXX...hls/)ディレクトリからの相対パス.
    // - 例: video/01.cmfv, audio/02.cmfa
    String filenameRel;
    File tmpCacheDir; // <- tmpCacheDir
    File chunkavDir; // <- div
    File file;

    // - 通信をここに溜める.
    // - 書き込み量がcontent-lengthに満ちたら復号を始める.
    // - 復号が不要な場合はそのまま本番ファイルに名前変更する.
    // - 復号に必要な情報が揃っていなければ、movieInfo.onGotAudioKey
    //   or onGotVideoKeyへ登録.
    // - そして鍵取得時にもう一度実行.
    File partFile;
    FileChannel partChannel = null;
    RandomAccessFile partRAF = null;
    File decryptedFile;

    boolean needDecrypt;
    Cipher decrypter = null;

    boolean errorOccurred;

    // - サーバーから予告されるコンテンツ長.
    // - 復号が必要な場合においては復号前のコンテンツ長.
    // - 復号が不必要な場合はコンテンツ長.
    long contentLength = 0;

    // - 通常のhttp 200受信の場合は0から受信した量を数える.
    // - partial content受信の場合はヘッダが示す開始位置から数える.
    // - この値がcontentLengthと一致したら全て受信した状態.
    int contentPos = 0;

    AV av;

    // - trueならshlsbid動画(珍しい). falseならhlsbid動画(普通).
    // - "/hlsbid/"となっているところが"/shlsbid/"となっているURL群で
    //   配信される動画がある(主にsoXXX).
    // - これらはレスポンスの仕様からして違う.
    // - このフラグはログ表示にのみ関わる.
    boolean shlsbid = false;

    public ChunkListener
    (DomandCVIEntry movieInfo, AV av, String filename, boolean needDecrypt
     , NLEventSource eventSource, Executor executor) {

        // TODO: errorOccuredを繰り返しifに入れるよりもreturnした方がいい.

        this.errorOccurred = av.isUnspecified();
        this.av = av;

        this.movieInfo = movieInfo;
        this.eventSource = eventSource;
        this.executor = executor;

        this.cache = movieInfo.getCache();

        this.tmpCacheDir = cache.getCacheTmpFile();

        // - 存在しない場合はnullが返る.
        // - ここで上記tmpCacheDirのテストもしている.
        this.chunkavDir = CmafCachingProcessor.getTmpStreamCmafavDirectory(
            cache, av.isAudio() ? "audio" : "video");

        if (chunkavDir == null) {
            // 途中でキャッシュコンプリートされているだけなら、ログしない.
            if (!cache.exists()) {
                Logger.info("error: chunkavDir==null: " + movieInfo.getSmid() + ": "
                            + filename + ": " + cache.getCacheFileName());
            };
            this.errorOccurred = true;
        };
        if (!errorOccurred) {
            if (!chunkavDir.exists()) {
                if (cache.exists()) {
                    // - 途中でキャッシュコンプリートされた.
                    // - エラー出さない.
                    // do nothing.
                    this.errorOccurred = true;
                }
                else if (!chunkavDir.mkdir()) {
                    Logger.info("error: mkdir failed: " + chunkavDir);
                    this.errorOccurred = true;
                };
            };
        };

        if (!errorOccurred) {
            this.file = new File(this.chunkavDir, filename);
            if (file.exists()) {
                // - このファイルはもうキャッシュする必要はないというフラグ.
                // - 本当はエラーじゃないからメッセージも出さない.
                this.errorOccurred = true;
            };
        };

        if (!errorOccurred) {
            // 個別キャッシュディレクトリからの相対パス.
            this.filenameRel = this.tmpCacheDir.toURI()
                .relativize(this.file.toURI()).toString();
        };

        // 既エラーではない場合のみログを出す.
        if (!errorOccurred && filenameRel.startsWith("file:")) {
            this.errorOccurred = true;
            Logger.info("-- 'file:'表現出現:tmpCacheDir:<"
                        + this.tmpCacheDir.toURI() + ">"
                        + ",file:<" + this.file.toURI() + ">"
                        + ",chunkavDir:<" + this.chunkavDir + ">");
        };

        // この一時ファイルが有効であるのは現在のjava仮想マシンが動いている
        // 間だけ. smXXX[]_titleフォルダ下よりも共通の一時ファイルディレクト
        // リ(を作ってそこ)に置いた方がゴミが残りにくいだろう.
        this.decryptedFile = new File(this.tmpCacheDir, "tmpcmfD_" + filename);
        this.decryptedFile.deleteOnExit(); // 期待しない.

        this.partFile = new File(this.tmpCacheDir, "tmpcmfP_" + filename);
        this.partFile.deleteOnExit(); // 期待しない.

        this.needDecrypt = needDecrypt;

        // Logger.info(String.format("--- cachedir: %s: %s", this.tmpCacheDir.exists() ? "exists" : "not found", this.tmpCacheDir));
    };

    // - 複数回呼び出される.
    // - decrypterが利用可能状態ならtrue.
    private boolean initDecrypter() {
        if (this.decrypter != null) {
            return true;
        };

        try {
            if (needDecrypt) {
                byte[] iv = getIV(movieInfo, av);
                byte[] key = getEncKey(movieInfo, av);
                this.decrypter = createDecrypter(iv, key, av);
            };
        } catch (NoSuchAlgorithmException e) {
            error(e); // 高確率でコーディングミス.
        } catch (NoSuchPaddingException e) {
            error(e); // 高確率でコーディングミス.
        } catch (InvalidAlgorithmParameterException e) {
            error(e); // 高確率でコーディングミス.
        } catch (InvalidKeyException e) {
            error(e); // 通信の化けか？
        };
        return this.decrypter != null;
    };

    private void error() {
        errorOccurred = true;
        close(false);
    };

    private void error(Exception e) {
        errorOccurred = true;
        close(false);
        Logger.error(e);
    };

    private void error(String msg) {
        errorOccurred = true;
        close(false);
        Logger.info(msg);
    };

    private void errorwarning(String msg) {
        errorOccurred = true;
        close(false);
        Logger.warning(msg);
    };

    private void close(boolean deletePart) {
        CloseUtil.close(partChannel);
        CloseUtil.close(partRAF);
        partChannel = null;
        partRAF = null;

        if (deletePart) {
            if (partFile.exists()) {
                partFile.delete();
            };
        };
    };

    private static byte[] getIV(DomandCVIEntry movieInfo, AV av) {
        if (av.isAudio()) {
            return movieInfo.getAudioIV();
        } else if (av.isVideo()) {
            return movieInfo.getVideoIV();
        };
        return null;
    };

    private static byte[] getEncKey(DomandCVIEntry movieInfo, AV av) {
        if (av.isAudio()) {
            return movieInfo.getAudioKey();
        } else if (av.isVideo()) {
            return movieInfo.getVideoKey();
        };
        return null;
    };

    private static Cipher createDecrypter
    (byte[] ivbytes, byte[] keybytes, AV av)
        throws NoSuchAlgorithmException, NoSuchPaddingException
        , InvalidKeyException, InvalidAlgorithmParameterException
    {
        if (av == AV.UNSPECIFIED) {
            return null; // TODO: nullよりも例外を飛ばすべき.
        };
        if (ivbytes == null || keybytes == null) {
            return null;
        };

        // - [RFC 8216 HTTP Live Streaming(HLS)]ではPKCS7を求めている.
        // - このPKCS5Paddingは実質PKCS7だから誤りではない.
        Cipher decrypter = Cipher.getInstance("AES/CBC/PKCS5Padding");

        IvParameterSpec iv = new IvParameterSpec(ivbytes);
        SecretKeySpec key = new SecretKeySpec(keybytes, "AES");

        decrypter.init(Cipher.DECRYPT_MODE, key, iv);
        // InvalidKeyException, InvalidAlgorithmParameterException

        return decrypter;
    };

    // "bytes 10-20/30"ならreturn [10,20,30].
    private int[] parseSingleContentRange
    (List<String> contentRangeList) {

        if (contentRangeList.size() == 0) {
            errorwarning("不明なpartial contentです(0): " + filenameRel);
            return null;
        };

        if (contentRangeList.size() != 1) {
            errorwarning(
                "<206 multipart partial content> was responded: "
                + filenameRel + ": '" + contentRangeList + "'");
            return null;
        };

        String contentRange = contentRangeList.get(0);
        if (!contentRange.startsWith("bytes ")) {
            errorwarning("不明なpartial contentです(1): " + filenameRel
                         + "'" + contentRange + "'");
            return null;
        };

        // "bytes 10-20/30" → "10-20/30"
        String ablstr = contentRange.substring("bytes ".length());
        // "10-20/30" → "10-20", "30"
        String[] ab_l = ablstr.split("/");
        if (ab_l.length != 2) {
            errorwarning("不明なpartial contentです(2): " + filenameRel
                         + "'" + contentRange + "'");
            return null;
        };

        // "10-20" → "10", "20"
        String[] a_b = ab_l[0].split("-");
        if (a_b.length != 2) {
            errorwarning("不明なpartial contentです(3): " + filenameRel
                         + "'" + contentRange + "'");
            return null;
        };

        try {
            int a = Integer.parseInt(a_b[0]);
            int b = Integer.parseInt(a_b[1]);
            int l = Integer.parseInt(ab_l[1]);
            return new int[]{a, b, l};
        } catch (NumberFormatException e) {
            Logger.error(e);
        };
        errorwarning("不明なpartial contentです(4): " + filenameRel
                     + ": '" + contentRange + "'");
        return null;
    };

    private void prepareForPartialContent(HttpResponseHeader responseHeader) {

        List<String> contentRanges =
            responseHeader.getMessageHeadersOfName("Content-Range");
        int[] abl = parseSingleContentRange(contentRanges);

        if (abl == null) {
            // エラーはもう出している.
            return;
        };

        contentLength = abl[2];
        int start = abl[0];
        int end = abl[1] + 1; // +1で開区間を閉区間へ.

        openPartFileForWrite();

        try {
            if (start >= partRAF.length()) {
                errorwarning(
                    "未取得位置から開始するpartial contentです: "
                    + filenameRel + ": 取得済み[" + partRAF.length()
                    + "] Content-Range[" + contentRanges.get(0) + "]");
                return;
            };
        } catch (IOException e) {
            errorwarning("ファイルサイズ取得失敗: " + partFile);
            return;
        };

        try {
            partRAF.seek(start);
            contentPos = start;
        } catch (IOException e) {
            errorwarning("seek失敗: " + partFile);
            return;
        };

        Logger.info(filenameRel + ": 再開 " + start + "-" + end);
    };

    private void openPartFileForWrite() {
        if (partRAF != null || partChannel != null) {
            errorwarning("error: partRAF!=null||partChannel!=null: "
                         + "コーディングミス");
            return;
        };

        try {
            partRAF = new RandomAccessFile(partFile, "rw");
        } catch (FileNotFoundException e) {
            errorwarning("書き込みopen失敗: " + partFile);
            return;
        };
        partChannel = partRAF.getChannel();
    };

    private void decryptAsync() {
        // ダウンロードした動画チャンクの復号処理を非同期で行なう.
        if (initDecrypter()) {
            // 復号情報の用意完了. run()呼び出し.
            executor.execute(this);
        } else {
            // まだ復号情報の用意が出来ていない. run()呼び出しを予約.
            Runnable async = new AsyncExecute(this, executor);
            if (av.isAudio()) {
                movieInfo.addGotAudioDecryptInfoListeners(async);
            } else {
                movieInfo.addGotVideoDecryptInfoListeners(async);
            };
        };
    };

    // - decryptしてその結果をfileへ.
    // - DL中フラグを下げる.
    // - 必要ならキャッシュコンプリート処理.
    @Override
    public void run() {
        // - この処理はRunnableとしてmovieInfo.gotAudioDecryptInfoListenersあるいは
        //   gotVideoDecryptInfoListeners経由で呼ばれる. 呼び出し機序はDomandCVIEntry.javaを参照.
        // - あるいはexecutor.execute経由で呼ばれる.

        // Logger.info("run on thread[" + Thread.currentThread().getId() + "]");
        if (errorOccurred) {
            return;
        };

        if (decrypter == null) {
            if (!initDecrypter()) {
                Cache.decrementDL(movieInfo.getVideoDescriptor());
                close(false);
                Logger.info(filenameRel + ": 復号準備失敗");
                return;
            };
        };

        decrypt();

        if (errorOccurred) {
            Cache.decrementDL(movieInfo.getVideoDescriptor());
            close(false);
            return;
        };

        partFile.renameTo(file);
        if (!file.exists()) {
            errorwarning("移動失敗(pf): '" + partFile + "' → '"+ file + "'");
            Cache.decrementDL(movieInfo.getVideoDescriptor());
            close(false);
            return;
        };
        partFile.delete();

        mightEndCache();
    };

    // - partFileの中身を復号してtmpFileへ.
    // - 復号失敗時はpartFileもtmpFileも削除.
    // - partRAF,partChannelはcloseされていること.
    // - partFileのサイズチェックはされていること.
    private void decrypt() {

        FileInputStream partIStream;
        FileChannel partIChannel;

        FileChannel decryptedChannel = null; // <- cacheChannel
        FileOutputStream decryptedOStream = null;

        // - IllegalBlockSizeExceptionが起きた時のためにそれを表示するため
        //   だけにdecrypterに入力した暗号文サイズを数える。
        int decrypterInputCounter = 0;

        byte[] rawbuf = new byte[1024 * 4]; // 効率良さげなサイズ.
        ByteBuffer bytebuf = ByteBuffer.wrap(rawbuf);

        try {
            partIStream = new FileInputStream(partFile);
            partIChannel = partIStream.getChannel();
        }
        catch (FileNotFoundException e) {
            // 成果ファイルが既にあるならログしない.
            if (file.exists() || !movieInfo.getCacheSaveFlag()) {
                // do nothing.
            } else {
                error(e);
            };
            return;
        };

        try {
            if (partIStream.available() != contentLength) {
                String s = String.format(
                    "%s: データ長不一致;コーディングミス. "
                    + "content-length[%d] ファイル長[%d]"
                    , filenameRel, contentLength, decrypterInputCounter);
                errorwarning(s);
            };
        } catch (IOException e) {
            errorwarning("入力ファイルチェック失敗");
            CloseUtil.close(partIChannel);
            CloseUtil.close(partIStream);
            return;
        };

        try {
            decryptedOStream = new FileOutputStream(decryptedFile);
            decryptedChannel = decryptedOStream.getChannel();
        } catch (FileNotFoundException e) {
            error(e);
            close(false); // partファイルの削除はしない.
            CloseUtil.close(partIChannel);
            CloseUtil.close(partIStream);
            return;
        };

        // partから読み取って復号してdecryptedへ書き込むループ.
        try {
            long reads = 0;
            for (;;) {
                reads = partIChannel.read(bytebuf);
                if (reads <= 0) {
                    break;
                };
                byte[] deced = decrypter.update(rawbuf, 0, (int)reads);
                decryptedChannel.write(ByteBuffer.wrap(deced));

                decrypterInputCounter += (int)reads;
                bytebuf.clear();
            };
            CloseUtil.close(partIChannel);
            CloseUtil.close(partIStream);
            partFile.delete();
        } catch (IOException e) {
            error(e);
            close(true); // trueでpartファイルの削除もする.
            CloseUtil.close(partIChannel);
            CloseUtil.close(partIStream);
            CloseUtil.close(decryptedChannel);
            CloseUtil.close(decryptedOStream);
            return;
        };

        // TODO: 別プロセスが過剰に書き込み、decryptedFileがサイズ過剰だった
        ///      場合の判定をここに.

        // 復号終了処理.
        byte[] deced;
        try {
            deced = decrypter.doFinal();
        } catch (IllegalBlockSizeException e) {
            // - AES-128では入力総バイト数が16の倍数ではないという例外.
            String s = String.format(
                "%s: %s. content-length[%d%%16=%d] input-count[%d%%16=%d]"
                , filenameRel, "IllegalBlockSizeException"
                , contentLength, contentLength % 16
                , decrypterInputCounter, decrypterInputCounter % 16);
            errorwarning(s);
            CloseUtil.close(partIChannel);
            CloseUtil.close(partIStream);
            CloseUtil.close(decryptedChannel);
            CloseUtil.close(decryptedOStream);
            return;
        } catch (BadPaddingException e) {
            errorwarning(file + ": " + e.toString());
            CloseUtil.close(partIChannel);
            CloseUtil.close(partIStream);
            CloseUtil.close(decryptedChannel);
            CloseUtil.close(decryptedOStream);
            return;
        };

        try {
            decryptedChannel.write(ByteBuffer.wrap(deced));
        } catch (IOException e) {
            errorwarning(file + ": " + e.toString());
            CloseUtil.close(partIChannel);
            CloseUtil.close(partIStream);
            CloseUtil.close(decryptedChannel);
            CloseUtil.close(decryptedOStream);
            return;
        };

        if (CloseUtil.close(decryptedChannel) &&
            CloseUtil.close(decryptedOStream)) {
            // do nothing
        } else {
            errorwarning(decryptedFile + ": close失敗");
        };

        decryptedFile.renameTo(file);
        if (!file.exists()) {
            errorwarning("移動失敗(df): '" + decryptedFile + "' → '"+ file
                         + "'");
            return;
        };
        decryptedFile.delete();
    };

    // - close成功でtrue.
    // - 両者null時もtrue.
    private boolean closePartFile() {
        boolean result =
            CloseUtil.close(partChannel) &&
            CloseUtil.close(partRAF);
        partChannel = null;
        partRAF = null;
        return result;
    };

    // - debugとメッセージ用.
    private boolean isFirstCmfv() {
        return filenameRel.contains("/1.cmfv")
            || filenameRel.contains("/01.cmfv")
            || filenameRel.contains("/001.cmfv")
            || filenameRel.contains("/0001.cmfv")
            || filenameRel.contains("/00001.cmfv");
    };

    public void setShlsbid(boolean x) {
        shlsbid = x;
    };

    @Override
    public void onResponseHeader(HttpResponseHeader responseHeader) {

        if (errorOccurred) {
            return;
        };

        int statusCode = responseHeader.getStatusCode();

        // - Content-Lengthは通信のボディサイズを表すが、contentLength
        //   プロパティは完成形のファイルサイズを表す.
        // - そのため206ではこの値は変更される.
        contentLength = responseHeader.getContentLength();

        if (statusCode == 304) {
            // ファイルが既にあるなら通知しない.
            if (! file.exists()) {
                errorwarning("<304 not modified>応答: "
                             + filenameRel);
            };
            // Logger.info(responseHeader.toString());
            return;
        };

        if (contentLength == -1) {
            // - contentLength==-1をshlsbid動画のフラグとして使う.
            // - shlsbid動画の場合に、Content-Lengthがないヘッダが送られて
            //   来る.
            // - shlsbid動画は206が来ない.
            // - ログだけ出して非エラーしておく.
            if (!shlsbid) {
                Logger.info("no content-length header: " + filenameRel);
            };
        } else if (contentLength < 0) {
            Logger.info("Invalid Content-Length[" + contentLength + "]: "
                        + filenameRel);
        };

        if (statusCode == 206) {
            // - partial content
            // - この経路ではpartFileを事前に削除しない.
            prepareForPartialContent(responseHeader);
            return;
        };

        if (statusCode != 200) {
            errorwarning("Invalid status code[" + statusCode + "]: " + filenameRel);
            return;
        };

        openPartFileForWrite();
    };

    @Override
    public void onTransferBegin(OutputStream receiverOut) {
        // do nothing
    };

    @Override
    public void onTransferring(byte[] input, int length) {
        if (errorOccurred) {
            return;
        };

        ByteBuffer bb = ByteBuffer.wrap(input, 0, length);
        try {
            partChannel.write(bb);
            contentPos += length;
        } catch (IOException e) {
            errorwarning(filenameRel + ": 書き込み失敗");
        };
        return;
    };

    @Override
    public void onTransferEnd(boolean completed) {
        if (errorOccurred) {
            // - hls版からコメント転記.
            // - [nl] DL中フラグを消す.
            // - Cache#store()後じゃないとExtension等に不具合が出るので注意.
            // - Cache#store()でVideoDescriptorが差し替わることに注意.
            // - 差し替わり前のvideoDescriptorを対象にdecrementする.
            Cache.decrementDL(movieInfo.getVideoDescriptor());
            close(false);
            return;
        };

        if (!closePartFile()) {
            errorwarning(file + ": partFile close失敗");
            Cache.decrementDL(movieInfo.getVideoDescriptor());
            close(false);
            return;
        };

        if (completed && contentLength == -1) {
            // - contentLength==-1をshlsbid動画のフラグとして使う.
            // - content-lengthなしで送られて来ていて、なおかつ今completed
            //   ならば、それを信頼する.
            contentLength = contentPos;
        }
        else if (contentPos > contentLength) {
            String s = String.format(
                "%s: 過剰なコンテンツボディ: 予告[%d], 応答[%d]"
                , filenameRel, contentLength, contentPos);
            errorwarning(s);
            Cache.decrementDL(movieInfo.getVideoDescriptor());
            close(false);
            return;
        }
        else if (contentPos != contentLength) {
            String s = String.format(
                "%s: 未完 %d/%d", filenameRel, contentPos, contentLength);
            errorwarning(s);
            Cache.decrementDL(movieInfo.getVideoDescriptor());
            close(false);
            return;
        };

        if (needDecrypt) {
            decryptAsync();
            return;
        };

        // 別スレッドが上書きした可能性も想定して、existsによる確認をする.
        partFile.renameTo(file);
        if (!file.exists()) {
            errorwarning("移動失敗(ppf): '" + partFile + "' → '"+ file + "'");
            Cache.decrementDL(movieInfo.getVideoDescriptor());
            close(false);
            return;
        };
        partFile.delete();

        mightEndCache();
    };

    private void mightEndCache() {

        if (Boolean.getBoolean("showCaching")) {
            Logger.info("caching " + movieInfo.getSmid());
        };

        CacheManager.HlsTmpSegments hlsTmpSegments;
        synchronized (movieInfo) {
            hlsTmpSegments = movieInfo.getHlsTmpSegments();
            if (hlsTmpSegments == null) {
                // ここでmaster.m3u8からファイルを辿り必要なファイル一覧を算出する.
                CacheManager.HlsTmpSegments x =
                    CacheManager.HlsTmpSegments.get(movieInfo.getVideoDescriptor());
                if (x == null) {
                    errorwarning("(cmaf|mec)hlsTmpSegments==null: "
                                  + movieInfo.getVideoDescriptor().toString());
                    movieInfo.setCacheSaveFlag(false);
                    return;
                };
                movieInfo.setHlsTmpSegments(x);
                hlsTmpSegments = x;
            }
            else {
                hlsTmpSegments.addCachedSegment(filenameRel);
            };

            // if (hlsTmpSegments.undownloadedKnownSegmentsCount() == 0) {
            //     Logger.info("--" + movieInfo.getVideoDescriptor()
            //                 + ": hlsTmpSegments.undownloadedKnownSegmentsCount()==0");
            // };

            movieInfo.mightSegmentsComplete();
        };
    };

};
// End Class ChunkListener.


enum AV {
    AUDIO, VIDEO, UNSPECIFIED;
    public boolean isAudio() {
        return this == AUDIO;
    };
    public boolean isVideo() {
        return this == VIDEO;
    };
    public boolean isUnspecified() {
        return this == UNSPECIFIED;
    };
    public String toString() {
        if (isAudio()) {
            return "audio";
        };
        if (isVideo()) {
            return "video";
        };
        return "unspecified";
    };
};


@SuppressWarnings("serial")
class NoIdInfoException extends Exception {
};


class DomandCVIUtil {

    // Hlsで言うMovieData(情報コンテナ)と同じ役割を持つ共有エントリーを作る.
    public static DomandCVIEntry initAndPutEntry(
        String entryKeyName
        , String videoType, String videoNumId, String videoHeightText
        , String audioKbpsText , String videoMode, String videoSrcId
        , String audioSrcId)
        throws NoIdInfoException, NumberFormatException {

        int videoHeight = Integer.parseInt(videoHeightText);
        int audioKbps = Integer.parseInt(audioKbpsText);

        String postfix = Cache.HLS;

        NicoIdInfoCache.Entry idInfo = NicoIdInfoCache.getInstance().get(videoNumId);
        if (idInfo == null) {
            throw new NoIdInfoException();
        };

        boolean lowAccess = getLowAccess(idInfo, videoSrcId, audioSrcId);

        // - dmcLow(360p-lowestのようなもの. これは360pとは違う.)かどうか
        //   はVideoDescriptorのコンストラクタ内でvideoModeから判定される.
        VideoDescriptor videoDescriptor = getVideoDescriptor(
            videoType + videoNumId, postfix, lowAccess, videoMode, audioKbps
            , /*srcid*/"");

        Cache cache = getCache(idInfo, videoDescriptor, lowAccess);

        DomandCVIEntry entry = new DomandCVIEntry(
            entryKeyName, videoType, videoNumId, videoHeight, audioKbps
            , videoMode, videoSrcId, audioSrcId, lowAccess, postfix
            , idInfo, videoDescriptor, cache);

        NLShared.INSTANCE.getDomandCVIManager().update(entry);

        return entry;
    };

    // - lowとは最高画質ではないか最高音質ではないということ.
    // - videoSrcIdがnullである場合、画質がlowであるかどうかを判定しない.
    // - audioSrcIdがnullである場合、音質がlowであるかどうかを判定しない.
    private static boolean getLowAccess(
        NicoIdInfoCache.Entry idInfo, String videoSrcId, String audioSrcId) {
        // DMCのメソッドを使っているのは誤りではない.
        if (videoSrcId != null) {
            Boolean low = idInfo.getDmcVideoEconomy(videoSrcId);
            if (null != low && low) {
                return true;
            };
        };
        if (audioSrcId != null) {
            Boolean low = idInfo.getDmcAudioEconomy(audioSrcId);
            if (null != low && low) {
                return true;
            };
        };
        return false;
    };

    private static VideoDescriptor getVideoDescriptor(
        String smid, String postfix, boolean lowAccess, String videoMode,
        int audioKbps, String srcId) {
        // - DMCのメソッドを使っているのは誤りではない.
        // - videoBitrate使用はDMCよりも前に廃止された.
        VideoDescriptor vd = VideoDescriptor.newDmc(
            smid, postfix, lowAccess, videoMode, /*videoBitrate*/0
            , audioKbps, srcId);
        VideoDescriptor regvd = Cache.getRegisteredVideoDescriptor(vd);
        if (regvd != null) {
            return regvd;
        };
        return vd;
    };

    private static Cache getCache(
        NicoIdInfoCache.Entry idInfo, VideoDescriptor vd
        , boolean lowAccess) {

        Cache cache;
        if (idInfo == null) {
            cache = new Cache(vd);
        }
        else {
            cache = new Cache(vd, idInfo.getTitle());
        };
        if (!lowAccess) {
            cache.unmarkLow();
        };
        return cache;
    };
};

/**
 * キャッシュ用データ管理クラス
 */
class DomandMovieData {
    private String smid; // sm,so,nm付きの動画番号ID.
    // HLS版のbitrateの単位はキロ. こちらではそれを明記する.
    private int audioKbps;
    private int videoHeight;
    private String postfix; // 拡張子
    private NicoIdInfoCache.Entry idInfo; // videoNumId と紐付いている情報.
    private VideoDescriptor videoDescriptor;
    private Cache cache;
    private String videoType; // sm,so,nmなど.
    private String videoNumId; // smなどを除いた動画番号.
    private boolean lowAccess; // 最上のvideoと最上のaudioならばfalse.
    private String videoMode; // 例: "1080p"
    // domand仕様に"360p_low"はない(2024-03).

    public DomandMovieData(
        String videoType, String videoNumId, String videoHeight, String audioKbps
        , String videoMode, String videoSrcId, String audioSrcId)
        throws NoIdInfoException, NumberFormatException {

        this.videoType = videoType;
        this.videoNumId = videoNumId;
        this.smid = videoType + videoNumId;
        initIdInfo(videoNumId);
        this.postfix = Cache.HLS;
        this.audioKbps = Integer.parseInt(audioKbps);
        this.videoHeight = Integer.parseInt(videoHeight);
        initLowAccess(videoSrcId, audioSrcId);
        this.videoMode = videoMode;
        initCache(this.idInfo, this.videoMode);
    };

    private void initIdInfo(String videoNumId) throws NoIdInfoException {
        this.idInfo = NicoIdInfoCache.getInstance().get(videoNumId);
        if (this.idInfo == null) {
            throw new NoIdInfoException();
        };
    };

    private void initLowAccess(String videoSrcId, String audioSrcId) {
        // 引数例: "video-h264-1080p", "audio-aac-128kbps"

        // DMCのメソッドを使っているのは誤りではない.
        Boolean videoLow = idInfo.getDmcVideoEconomy(videoSrcId);
        Boolean audioLow = idInfo.getDmcAudioEconomy(audioSrcId);
        if (videoLow == null && audioSrcId == null) {
            this.lowAccess = false;
            return;
        };
        this.lowAccess = videoLow || audioLow;
    };

    private void initCache(NicoIdInfoCache.Entry idInfo, String videoMode) {
        this.videoDescriptor = VideoDescriptor.newDmc(
            smid, postfix, lowAccess, videoMode,
            /*videoBitrate*/0, audioKbps, /*srcid*/"");
        VideoDescriptor regVideoDesc =
            Cache.getRegisteredVideoDescriptor(this.videoDescriptor);
        if (regVideoDesc != null) {
            this.videoDescriptor = regVideoDesc;
        };
        if (this.idInfo == null) {
            // HLSにならってこう書く. ここを通ることはあるか？
            cache = new Cache(videoDescriptor);
        }
        else {
            cache = new Cache(videoDescriptor, idInfo.getTitle());
        };
        if (!this.lowAccess) {
            cache.unmarkLow();
        };
    };

    public Cache getCache() { return cache;};
    public String getSmid() { return smid;};
    public VideoDescriptor getVideoDescriptor() { return videoDescriptor;};
    public String getPostfix() { return postfix;};
    public NicoIdInfoCache.Entry getIdInfo() { return idInfo;};
    public String getVideoType() { return videoType;};
    public String getVideoId() { return videoNumId;};
    public int getVideoHeight() {return videoHeight;};
};
