package dareka.common;

import java.util.ArrayList;
import java.util.function.UnaryOperator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class M3u8Util {

    // > RFC 8216 HTTP Live Streaming.
    // これに基くm3u8ファイルのユーティリティー.
    // > Blank lines are ignored.

    // - 改行を含まない1行だけの文字列からURI=""表現の中身を抽出する.
    // - 空のURI attributeにはマッチしない.
    private static final Pattern M3U8_URL_PATTERN = Pattern.compile(
        "(?<=[,:]URI=\")([^\"]+)(?=\")");

    // - m3u8表現のcontentからurl部を抜きだして、それをsfilterに渡す.
    //   その結果をcontentに戻す. 機能型文字列replace処理.
    // - sfilterが素通しラムダならreturnと引数のcontentはequalする.
    // - URI=""のような空URLはsfilterを通らない.
    public static String replaceURL(String content,
                                    UnaryOperator<String> sfilter) {
        StringBuilder sb = new StringBuilder(content.length() * 2);

        for (String line : content.split("\n")) {
            if (line.isEmpty()) {
                sb.append("\n");
                continue;
            };

            if (! line.startsWith("#")) {
                sb.append(sfilter.apply(line));
                sb.append("\n");
                continue;
            };

            if (! line.startsWith("#EXT")) {
                sb.append(line);
                sb.append("\n");
                continue;
            };

            Matcher m = M3U8_URL_PATTERN.matcher(line);
            if (! m.find()) {
                sb.append(line);
                sb.append("\n");
                continue;
            };

            String left = line.substring(0, m.start(1));
            String right = line.substring(m.end(1));
            sb.append(left);
            sb.append(sfilter.apply(m.group(1)));
            sb.append(right);
            sb.append("\n");
        };
        // int continuepos = 0;
        // Matcher m = M3U8_URL_PATTERN.matcher(content);
        // while (m.find(continuepos)) {
        //     int nomatchStart = continuepos;
        //     int nomatchEnd = m.start(1);
        //     continuepos = m.end(1);
        //     sb.append(content.substring(nomatchStart, nomatchEnd));
        //     sb.append(sfilter.apply(m.group(1)));
        // };
        // sb.append(content.substring(continuepos));
        return sb.toString();
    };

    // m3u8 contentに含まれるURLのsearch部に引数searchを追加する.
    public static String injectURLSearch(String content, String search) {
        return replaceURL(content, (url) -> {
                if (url.contains("?")) {
                    return url + "&" + search;
                };
                return url + "?" + search;
            });
    };

    public static ArrayList<String> getUrlList(String content) {

        ArrayList<String> a = new ArrayList<>();

        for (String line : content.split("\n")) {
            if (line.isEmpty()) {
                continue;
            };

            if (! line.startsWith("#")) {
                a.add(line);
                continue;
            };

            if (! line.startsWith("#EXT")) {
                continue;
            };

            Matcher m = M3U8_URL_PATTERN.matcher(line);
            if (! m.find()) {
                continue;
            };

            a.add(m.group(1));
        };
        return a;
    };
};
