package dareka.processor.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.UnsupportedOperationException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import dareka.common.Logger;
import dareka.extensions.CompleteCache;
import dareka.extensions.SystemEventListener;

// - Domand仕様の通信間で情報を共有するためのコンテナ.
// - nltmp_smXXX[xxxp,xxx]_title.hlsを作成する前の状態からそれをstore(complete)処理する
//   までの状態を扱う.
// - このクラスに実装寄りのコードを書くべきではないが現在含まれている(chunkLoadStart,
//   mightWrite, segmentsComplete, cacheStoreなど).
// - audio用とvideo用それぞれのプロパティが含まれるのは名残りであって、現在は一つの
//   サブプレイリストを扱うから整理するべき.
// - key値の仕様はロジック側(CmafCachingProcessor.java)を参照.
// - スペルアウトはDomand Cache Video Info Entry.
// - NLShared.INSTANCE.getDomandCVIManager().update(entry);
public class DomandCVIEntry {
    // - synchronizedの付け方に一貫性がない. 利用例に依存して付けている.

    // DomandCVIManagerがこのEntryを残すか捨てるか判断するのに使う.
    private long updatedOn;

    // 連想配列のキー(暗号鍵ではない). 重複しなければ形式は自由.
    // キャッシュ中は[動画IDと品質モード]とkeyが対応付いていると都合が良い.
    private final String key;

    private final String videoType; // smとか.
    private final String videoNumId; // smなどを除いた動画番号.
    private final int videoHeight; // 例: 1080
    private final int audioKbps; // 例: 128
    private final String videoMode; // 例: "1080p"
    private final String videoSrcId; // 例: "video-h264-1080p"
    private final String audioSrcId; // 例: "audio-aac-128kbps"
    private final Boolean lowAccess; // 最高品質の映像と音声であるならfalse.

    private final String postfix; // 例: ".hls"
    private final NicoIdInfoCache.Entry idInfo; // videoNumId と紐付いている情報.
    private final VideoDescriptor videoDescriptor;
    // videoDescriptorが示す品質よりも高品質なキャッシュを指す場合もある.
    private Cache cache;

    private Boolean cacheSaveFlag = true; // キャッシュしないならfalse.
    // 実装時点ではAES-128, AES/CBC/NoPaddingしか扱わないからmethodは記録しない.
    private byte[] audioIV = null; // 復号に使う初期化ベクトル
    private byte[] videoIV = null; // 復号に使う初期化ベクトル
    private byte[] audioKey = null; // 復号に使うキー
    private byte[] videoKey = null; // 復号に使うキー

    // - 復号に使う初期化ベクトルとキーは復号対象よりも後にダウンロードされるかも知れない.
    //   それに備えて復号処理を後で行なえるようにする.
    // - いまのところは順序が重要な処理は担わせないからHashSet.
    private Set<Runnable> gotAudioDecryptInfoListeners = new HashSet<>();
    private Set<Runnable> gotVideoDecryptInfoListeners = new HashSet<>();

    // - 動画チャンクをダウンロードし始めてからキャッシュ一時ディレクトリを作るようにするための
    //   変数たち.
    // - m3u8だけが読み込まれチャンクが読み込まれない状況を想定している.
    //   例えばvideo-h264-360p.m3u8が通信に上がった段階で一時ディレクトリを作ってしまうと
    //   m3u8だけが入ったnltmp_*ディレクトリが増殖してしまう.
    // - あまり複雑になったら分離すること. 分離を前提に組むこと.
    private boolean chunkLoadStartedFlag = false;
    // - ファイル名stringとbyte[]コンテンツの2要素を持つ配列にしてしまった方がいい.
    private byte[] masterM3u8 = null; // master.m3u8 書き込み待ちのファイル内容
    private byte[] videoM3u8 = null; // video.m3u8 上同
    private byte[] audioM3u8 = null; // audio.m3u8 上同

    // - ニコ動のcmaf動画は一つのaudioに複数のvideoが紐付いている.
    // - このプロパティはその相互の紐付けを管理する.
    // - 従って潜るように参照していくと循環する(同じインスタンスに繰り返し出会う).
    // - audioを管理するthisの場合、このプロパティにはvideoを管理するDomandCVIEntryが入る.
    // - videoの場合はaudioのそれが入る.
    // - このクラスは継承しないし、このプロパティの変更に伴う処理もないから
    //   setter/getterを用意しない.
    // - 5は1080p, 720p, 480p, 360p, 360p-lowestという想定.
    public List<DomandCVIEntry> assocList = new ArrayList<>(5);

    // - 初期化はロジック側が行う.
    private CacheManager.HlsTmpSegments hlsTmpSegments = null;

    // - trueならばthisが役割を終えた状態.
    //   - ハンドルしていたnltmpはすでに存在していない.
    private boolean completedFlag = false;

    public DomandCVIEntry getDomandCVIEntryByVideoSrcId(String videoSrcId) {
        for (DomandCVIEntry x : assocList) {
            if (videoSrcId.equals(x.videoSrcId)) {
                return x;
            };
        };
        return null;
    };

    public DomandCVIEntry(
        String key
        , String videoType, String videoNumId, int videoHeight, int audioKbps
        , String videoMode, String videoSrcId, String audioSrcId
        , Boolean lowAccess, String postfix, NicoIdInfoCache.Entry idInfo
        , VideoDescriptor videoDescriptor, Cache cache) {

        this.updatedOn = System.currentTimeMillis();

        this.key = key;

        this.videoType = videoType;
        this.videoNumId = videoNumId;
        this.videoHeight = videoHeight;
        this.audioKbps = audioKbps;
        this.videoMode = videoMode;
        this.videoSrcId = videoSrcId;
        this.audioSrcId = audioSrcId;
        this.lowAccess = lowAccess;

        this.postfix = postfix;
        this.idInfo = idInfo;
        this.videoDescriptor = videoDescriptor;
        this.cache = cache;
    };

    public long getUpdatedOn() {
        return updatedOn;
    };

    public String getKey() {
        return key;
    };

    public String getVideoType() {
        return videoType;
    };

    public String getVideoNumId() {
        return videoNumId;
    };

    // "smXXX" を返す
    public String getSmid() {
        // 表記先例はgetSmidが優勢. getSMIDもある.
        return videoType + videoNumId;
    };

    public int getVideoHeight() {
        return videoHeight;
    };

    public int getAudioKbps() {
        return audioKbps;
    };

    public String getVideoMode() {
        return videoMode;
    };

    public String getVideoSrcId() {
        return videoSrcId;
    };

    public String getAudioSrcId() {
        return audioSrcId;
    };

    public Boolean getLowAccess() {
        return lowAccess;
    };

    public String getPostfix() {
        return postfix;
    };

    public NicoIdInfoCache.Entry getIdInfo() {
        return idInfo;
    };

    public VideoDescriptor getVideoDescriptor() {
        return videoDescriptor;
    };

    public void setCache(Cache v) {
        cache = v;
    };

    public Cache getCache() {
        return cache;
    };

    public synchronized void restart() {
        cacheSaveFlag = true; // assocListに伝播させない.
        hlsTmpSegments = null;
        completedFlag = false;

        if (!isAudioHandling() && cache != null) {
            File nltmp = cache.getCacheTmpFile();
            // - nltmp状態のままaudioがあるのはおかしい.
            File audiom3u8 = new File(nltmp, "audio.m3u8");
            audiom3u8.delete();
        };

        if (videoDescriptor != null) {
            hlsTmpSegments = CacheManager.HlsTmpSegments.get(getVideoDescriptor());
            mightSegmentsComplete();
        };
    };

    public synchronized Boolean setCacheSaveFlag(Boolean v) {
        cacheSaveFlag = v;
        for (int i = assocList.size() - 1; i >= 0; --i) {
            DomandCVIEntry x = assocList.get(i);
            // assocListは循環参照する. 無限ループにならないように比較.
            if (x.getCacheSaveFlag() != v) {
                x.setCacheSaveFlag(v);
            };
        };
        return v;
    };

    // - thisが役割終了している場合もfalse.
    // - completedFlagを加味することは意味論上での一貫性が怪しいから要改善.
    public synchronized Boolean getCacheSaveFlag() {
        return !completedFlag && cacheSaveFlag;
    };

    private synchronized void mightCallAudioListeners() {
        if (audioIV == null || audioKey == null) {
            return;
        };
        for (Runnable run : gotAudioDecryptInfoListeners) {
            run.run();
        };
        gotAudioDecryptInfoListeners.clear();
    };

    private synchronized void mightCallVideoListeners() {
        if (videoIV == null || videoKey == null) {
            return;
        };
        for (Runnable run : gotVideoDecryptInfoListeners) {
            run.run();
        };
        gotVideoDecryptInfoListeners.clear();
    };

    // 登録した処理は最大1回だけ実行される.
    public synchronized void addGotAudioDecryptInfoListeners(Runnable r) {
        gotAudioDecryptInfoListeners.add(r);
    };

    // 登録した処理は最大1回だけ実行される.
    public synchronized void addGotVideoDecryptInfoListeners(Runnable r) {
        gotVideoDecryptInfoListeners.add(r);
    };

    public synchronized void setAudioIV(byte[] v) {
        audioIV = v;
        mightCallAudioListeners();
    };

    public synchronized void setVideoIV(byte[] v) {
        videoIV = v;
        mightCallVideoListeners();
    };

    public synchronized void setAudioKey(byte[] v) {
        audioKey = v;
        mightCallAudioListeners();
    };

    public synchronized void setVideoKey(byte[] v) {
        videoKey = v;
        mightCallVideoListeners();
    };

    public synchronized byte[] getAudioIV() {
        return audioIV;
    };

    public synchronized byte[] getVideoIV() {
        return videoIV;
    };

    public synchronized byte[] getAudioKey() {
        return audioKey;
    };

    public synchronized byte[] getVideoKey() {
        return videoKey;
    };

    public synchronized boolean getCompletedFlag() {
        return completedFlag;
    };

    public synchronized CacheManager.HlsTmpSegments getHlsTmpSegments() {
        return hlsTmpSegments;
    };

    public synchronized void setHlsTmpSegments(CacheManager.HlsTmpSegments v) {
        hlsTmpSegments = v;
    };

    public boolean isAudioHandling() {
        return videoSrcId == null;
    };

    public synchronized boolean isSegmentsComplete() {
        if (hlsTmpSegments == null) {
            return false;
        };
        return hlsTmpSegments.isKnownSegmentsComplete();
    };

    public synchronized void mightSegmentsComplete() {
        if (hlsTmpSegments == null) {
            return;
        };
        if (hlsTmpSegments.isKnownSegmentsComplete()) {
            segmentsComplete();
        };
    };

    public synchronized void segmentsComplete() {
        // - thisが扱っているsub playlist下のsegment(chunk)が揃った時に呼び出される.

        if (isAudioHandling()) {
            // - thisがaudio側ならば関連videoを試しcompleteを促す.

            // Logger.info("--audio complete: " + videoDescriptor);

            for (DomandCVIEntry video : assocList) {
                synchronized (video) {
                    if (video.isSegmentsComplete()) {
                        // Logger.info("--video complete: " + video.videoDescriptor);
                        video.videoSegmentsComplete();
                    }
                    else {
                        // do nothing.
                        // Logger.info("--video incomplete: " + video.videoDescriptor);
                    };
                };
            };
            return;
        };

        videoSegmentsComplete();
    };

    // - video-segmentsはcompleteしたという宣言的な名前.
    // - 必要に応じてcache complete処理とそれに必要な前処理をする.
    private synchronized void videoSegmentsComplete() {
        if (!getCacheSaveFlag()) {
            // Logger.info("--!getCacheSaveFlag: " + videoDescriptor);
            return;
        };
        if (!isSegmentsComplete()) {
            // Logger.info("--!isSegmentsComplete: " + videoDescriptor);
            return;
        };

        if (moveAudioToMeFromAudioNltmp()) {
            cacheStore();
            return;
        };

        if (copyAudioToMeFromAnotherCache()) {
            cacheStore();
            return;
        };
    };

    // - 失敗でfalse.
    // - キャッシュ中であるnltmp_smX[0p,XXX]_title.hlsからaudioデータを自キャッシュへ
    //   移動する.
    private synchronized boolean moveAudioToMeFromAudioNltmp() {
        DomandCVIEntry audioMovInfo = assocList.get(0);
        boolean success = true;
        synchronized (audioMovInfo) {
            if (audioMovInfo.completedFlag) {
                // audioMovInfoは使い終えた後の状態.
                // Logger.info("--audioMovInfo.completedFlag: " + videoDescriptor);
                return false;
            };
            if (!audioMovInfo.isSegmentsComplete()) {
                // Logger.info("--!audioMovInfo.isSegmentsComplete: " + videoDescriptor);
                return false;
            };

            // - audioMovInfo側がキャッシュコンプリートしている可能性を加味しない.
            // - audio側がコンプリートするようなコードは書いていない.

            Path src = audioMovInfo.getCache().getCacheTmpPath();
            Path dest = getCache().getCacheTmpPath();

            if (src == null || dest == null) {
                // 2024-09-08起きる.
                Logger.info("(dcvie|matmfan)programming error(3): "
                            + audioMovInfo.getVideoDescriptor() + " <- "
                            + getVideoDescriptor() + ": src=" + src
                            + ": dest=" + dest);
                return false;
            };

            try {
                Files.move(src.resolve("audio.m3u8"), dest.resolve("audio.m3u8")
                           , StandardCopyOption.REPLACE_EXISTING);
            }
            catch (AtomicMoveNotSupportedException
                   | DirectoryNotEmptyException
                   | FileAlreadyExistsException
                   | UnsupportedOperationException e) {
                Logger.info("(dcvie|matmfan)programming error(1): " + e.getClass()
                            + ": " + getVideoDescriptor() + " <- "
                            + audioMovInfo.getVideoDescriptor());
                success = false;
            }
            catch (IOException e) {
                Logger.info("failed to move audio.m3u8: "
                            + getVideoDescriptor() + " <- "
                            + audioMovInfo.getVideoDescriptor());
                success = false;
            };

            try {
                overwriteMoveFiles(src.resolve("audio"), dest.resolve("audio")
                                   , StandardCopyOption.REPLACE_EXISTING);
            }
            catch (AtomicMoveNotSupportedException
                   | DirectoryNotEmptyException
                   | FileAlreadyExistsException
                   | UnsupportedOperationException e) {
                Logger.info("(dcvie|matmfan)programming error(2): " + e.getClass()
                            + ": " + getVideoDescriptor() + " <- "
                            + audioMovInfo.getVideoDescriptor());
                success = false;
            }
            catch (IOException e) {
                Logger.info("failed to move audio dir: "
                            + ": " + getVideoDescriptor() + " <- "
                            + audioMovInfo.getVideoDescriptor());
                success = false;
            };

            try {
                audioMovInfo.getCache().deleteTmp();
            }
            catch (IOException e) {
                Logger.info("failed to delete: "
                            + audioMovInfo.videoDescriptor.toString());
            };
            audioMovInfo.completedFlag = true;
        };
        return success;
    };

    private synchronized boolean copyAudioToMeFromAnotherCache() {
        VideoDescriptor audioVD = CacheManager.getCachedHlsVideoByAudioKbps(
            getSmid(), getAudioKbps());
        if (audioVD == null) {
            return false;
        };

        Cache audioCache = new Cache(audioVD);
        Path src = audioCache.getCachePath();
        Path dest = getCache().getCacheTmpPath();

        if (src == null || dest == null) {
            // 2024-09-08起きる.
            Logger.info("(dcvie|matmfan)programming error(3): "
                        + audioVD + " <- " + getVideoDescriptor()
                        + ": src=" + src + ": dest=" + dest);
            return false;
        };

        try {
            Files.copy(src.resolve("audio.m3u8"), dest.resolve("audio.m3u8")
                       , StandardCopyOption.REPLACE_EXISTING);
            copyFolder(src.resolve("audio"), dest.resolve("audio")
                       , StandardCopyOption.REPLACE_EXISTING);
        }
        catch (AtomicMoveNotSupportedException
               | DirectoryNotEmptyException
               | FileAlreadyExistsException
               | UnsupportedOperationException e) {
            Logger.info("(dcvie|catmfac)programming error: " + e.getClass()
                        + ": " + getVideoDescriptor() + " <- "
                        + audioVD);
        }
        catch (IOException e) {
            Logger.info("failed to copy audio data: src: "
                        + audioCache.getCacheFileName());
            return false;
        };
        return true;
    };

    // ユーティリティー関数. 適切な場所へ移すこと.
    // https://stackoverflow.com/a/50418060
    public void copyFolder(Path source, Path target, CopyOption... options)
        throws IOException {

        Files.walkFileTree(source, new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs)
                    throws IOException {
                Files.createDirectories(target.resolve(source.relativize(dir).toString()));
                return FileVisitResult.CONTINUE;
            }
            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs)
                    throws IOException {
                Files.copy(file, target.resolve(source.relativize(file).toString())
                           , options);
                return FileVisitResult.CONTINUE;
            }
        });
    };

    public void overwriteMoveFiles(Path source, Path target, CopyOption... options)
        throws IOException {

        Files.walkFileTree(source, new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs)
                    throws IOException {
                Files.createDirectories(target.resolve(source.relativize(dir).toString()));
                return FileVisitResult.CONTINUE;
            }
            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs)
                    throws IOException {
                Files.move(file, target.resolve(source.relativize(file).toString())
                           , options);
                return FileVisitResult.CONTINUE;
            }
        });
    };

    private synchronized boolean cacheStore() {

        {
            // - ソースコードが正常ならば本来このチェックは不要だが2024-09-06時点通る
            //   ことがあるためしばらく残す.
            CacheManager.HlsTmpSegments.forget(getVideoDescriptor());
            CacheManager.HlsTmpSegments x =
                CacheManager.HlsTmpSegments.get(getVideoDescriptor());
            if (!x.allFilesExist(this.cache.getCacheTmpFile())) {
                Logger.info("(dcvie|cache store)segment missing: "
                            + this.cache.getCacheFileName());
                x.debugDump();
                restart();
                return false;
            };
        };

        try {
            cache.store();
        } catch (IOException e) {
            Logger.debugWithThread(e);
        };

        // [nl] 存在チェックと完了時のイベント通知.
        if (cache.exists()) {
            this.completedFlag = true;
        }
        else {
            Logger.debug("completion failed: " + cache.getCacheFileName());
            return false;
        };

        for (CompleteCache entry : NLShared.INSTANCE.getCompleteEntries()) {
            try {
                if (entry.onComplete(cache)) {
                    break; // trueが返ってきた時点キャッシュは移動した.
                };
            } catch (Throwable t) {
                Logger.error(t); // エラーは無視して続行
            };
        };
        if (NLShared.INSTANCE.countSystemEventListeners() > 0) {
            NLShared.INSTANCE.notifySystemEvent(
                SystemEventListener.CACHE_COMPLETED
                , new NLEventSource(null, /*requestHeader*/null, cache)
                , false);
        };

        Logger.info("cache completed: " + cache.getCacheFileName());
        return true;
    };

    // - mightはcacheSaveFlagがtrueかつnltmpCreatedがtrueであればキャッシュファイルとして
    //   書き込むという意味.
    // - 条件が揃っていない場合は溜め込まれ、chunkLoadStart()で書き込まれる.
    public synchronized void mightWriteMasterM3u8(byte[] content) {
        masterM3u8 = content;
        mightWrite();
    };
    public synchronized void mightWriteVideoM3u8(byte[] content) {
        videoM3u8 = content;
        mightWrite();
    };
    public synchronized void mightWriteAudioM3u8(byte[] content) {
        audioM3u8 = content;
        mightWrite();
    };
    // どのチャンクであれ動画チャンク・音声チャンクへの要求をキャッチした時に呼び出す.
    public synchronized void chunkLoadStart() {
        updatedOn = System.currentTimeMillis();

        if (!chunkLoadStartedFlag) {
            chunkLoadStartedFlag = true;

            // - 元々はplaylist urlをハンドルする部分で出していたメッセージ.
            // - その元の方法では検出した全てのplaylistで表示されたため鬱陶しかった.
            //   (audio, 1080p, 720p, 480p, 360p, 144p, 低画質それぞれに対して表示が
            //    出てしまう).
            // - チャンクを読み込み初めてから出すようにしたのがこれ.
            Logger.info("(dcvie)no cache found: " + getCache().getCacheFileName());

            // - 複数回呼び出しても無害だけど無駄だからフラグが変わる時1回だけ呼び出す.
            // - この時点でmasterM3u8==nullである場合は存在しない.
            mightWrite();
        };
    };

    private synchronized void mightWrite() {
        if (!getCacheSaveFlag()) {
            return;
        };
        if (!chunkLoadStartedFlag) {
            return;
        };
        if (masterM3u8 != null) {
            writeM3u8("master.m3u8", masterM3u8);
            masterM3u8 = null;
        };
        if (videoM3u8 != null) {
            writeM3u8("video.m3u8", videoM3u8);
            videoM3u8 = null;
        };
        if (audioM3u8 != null) {
            writeM3u8("audio.m3u8", audioM3u8);
            audioM3u8 = null;
        };
    };

    private synchronized void writeM3u8(String filename, byte[] content) {
        try {
            // cacheが設定済みでなければ、ここに来ないことを前提にしている.
            File dir = cache.prepareTmpHlsDirectory();
            File playlist = new File(dir, filename);

            if (playlist.exists()) {
                // 既に存在するなら内容比較.
                byte[] prevContent = new byte[(int) playlist.length()];
                try (FileInputStream stream = new FileInputStream(playlist)) {
                    stream.read(prevContent);
                };
                if (!Arrays.equals(content, prevContent)) {
                    // - 可能性は:
                    //   - NicoCacheが知らない理由でファイルが書き換えられた.
                    //   - 「この動画は投稿者によって修正されました」.
                    //   - NicoCacheのプレイリスト加工アルゴリズムが変わった.
                    Logger.info("Playlist mismatch: " + playlist.getPath());
                    deleteContainedFilesWithoutM3u8(dir); // 動画・音声チャンク削除.
                };
            };

            // 改行コードをLFにするためバイナリIO
            try (FileOutputStream fos = new FileOutputStream(playlist, false)) {
                fos.write(content);
            }
            catch (IOException e) {
                Logger.info("Failed to write list: " + playlist.getPath());
            };
        }
        catch (IOException e) {
            // - prepareTmpHlsDirectoryが失敗した.
            // - 異常状態だからキャッシュ抑制.
            Logger.info("Failed to create nltmp dir: " + getSmid());
            setCacheSaveFlag(false);
        };
    };

    private static void deleteContainedFilesWithoutM3u8(File dir) {
        File[] files = dir.listFiles();
        if (files == null) {
            return;
        };
        for (File file : files) {
            if (!file.getName().endsWith(".m3u8")) {
                file.delete();
            };
        };
    };
};
